/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * __________________
 *
 *  Copyright 2021 Adobe
 *  All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe.
 **************************************************************************/

package com.day.cq.analytics.sitecatalyst.util;
import org.apache.commons.io.IOUtils;
import org.apache.http.HttpEntity;
import org.apache.http.HttpStatus;
import org.apache.http.StatusLine;
import org.apache.http.client.config.RequestConfig;
import org.apache.http.client.methods.*;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.osgi.services.HttpClientBuilderFactory;
import org.apache.http.util.EntityUtils;
import org.osgi.service.component.annotations.Component;
import org.osgi.service.component.annotations.Reference;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import com.adobe.granite.crypto.CryptoSupport;

@Component
public final class WebServiceImpl implements WebService {

    /**
     * API Content-Type header value
     */
    private static final String CONTENT_TYPE = "application/json";

    /**
     * Default connection timeout in milliseconds
     */
    private static final int DEFAULT_CONNECTION_TIMEOUT = 60000;

    /**
     * Default socket timeout in milliseconds
     */
    private static final int DEFAULT_SOCKET_TIMEOUT = 30000;

    private int connectionTimeout = DEFAULT_CONNECTION_TIMEOUT;

    private int socketTimeout = DEFAULT_SOCKET_TIMEOUT;


    @Reference
    private HttpClientBuilderFactory clientBuilderFactory;

    @Reference
    private CryptoSupport cryptoSupport;

    private Logger log = LoggerFactory.getLogger(getClass());


    private void addHeaders(HttpUriRequest request, AnalyticsConfig analyticsConfig)
            throws Exception {

        if (analyticsConfig == null) return;

        String token = analyticsConfig.getAccessToken();
        // unprotect token if necessary
        if (cryptoSupport.isProtected(token)) {
            token = cryptoSupport.unprotect(token);
        }

        request.addHeader("Content-Type", String.format(CONTENT_TYPE));
        request.addHeader("X-Api-Key", analyticsConfig.getApikey());
        request.addHeader("x-proxy-global-company-id", analyticsConfig.getCompanyId());
        request.addHeader("Authorization", "Bearer " + token);
    }


    public String request(HttpRequestBase request, AnalyticsConfig analyticsConfig) throws Exception {

        CloseableHttpClient httpClient = null;
        HttpUriRequest apiRequest = null;
        CloseableHttpResponse apiResponse = null;
        try {
            RequestConfig config = RequestConfig.custom()
                    .setConnectTimeout(connectionTimeout)
                    .setConnectionRequestTimeout(connectionTimeout)
                    .setSocketTimeout(socketTimeout)
                    .build();

            httpClient = clientBuilderFactory.newBuilder().setDefaultRequestConfig(config).build();

            apiRequest = request;
            addHeaders(apiRequest, analyticsConfig);

            logRequest(apiRequest, "Analytics API Request");

            apiResponse = httpClient.execute(apiRequest);
            HttpEntity entity = (apiResponse != null) ? apiResponse.getEntity() : null;
            StatusLine responseStatusLine = (apiResponse != null) ? apiResponse.getStatusLine() : null;

            int responseStatusCode = responseStatusLine != null ? responseStatusLine.getStatusCode() : -1;
            if (responseStatusCode == HttpStatus.SC_OK || responseStatusCode == HttpStatus.SC_CREATED) {
                if (entity == null) {
                    throw new Exception("Invalid response! Empty payload.");
                }
                return EntityUtils.toString(entity);
            } else {
                String apiResponseJson = entity != null ? EntityUtils.toString(entity) : "";
                String error = String.format("Unexpected response status code [%d] for request [%s].\n%s",
                        responseStatusCode, apiRequest.getURI(), apiResponseJson);
                if (log.isDebugEnabled()) {
                    log.debug(error);
                }
                return null;
            }
        } catch (Exception e) {
            throw new Exception("API request failed", e);
        } finally {
            IOUtils.closeQuietly(apiResponse);
            IOUtils.closeQuietly(httpClient);
        }
    }

    private void logRequest(HttpUriRequest request, String message) throws Exception {
        if (log.isDebugEnabled()) {
            log.debug("{}: {} {} {}", message,
                    request.getMethod(),
                    request.getURI().toString(),
                    (request instanceof HttpEntityEnclosingRequestBase) ? EntityUtils.toString(((HttpEntityEnclosingRequestBase) request).getEntity()) : ""
            );
        }
    }

}

