/*******************************************************************************
 * ADOBE CONFIDENTIAL
 * __________________
 *
 * Copyright 2017 Adobe Systems Incorporated
 * All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Systems Incorporated and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Systems Incorporated.
 ******************************************************************************/

package com.day.cq.analytics.testandtarget.mac;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import java.util.HashMap;
import java.util.Map;

import org.apache.commons.lang.StringUtils;
import org.osgi.annotation.versioning.ProviderType;

import com.day.cq.analytics.testandtarget.TargetConstants;

/**
 * Holds the Marketing Cloud Metadata which is attached to objects exported to Target. Not all objects require this
 * metadata, so for now it's only attached to offers and activities
 * 
 */
@ProviderType
public class MacMetadata {

    private final Map<String, Object> metadata = new HashMap<String, Object>();

    public MacMetadata() {

    }

    /**
     * Adds a Marketing Cloud Metadata attribute / value pair to this offer request. The following restrictions apply:
     * <ul>
     * <li>the attributeName or attributeValue must not exceed 250 chars in length</li>
     * <li>The maximum size of the MAC Metadata list is 15 items</li>
     * </ul>
     * This method throws a {@link RuntimeException} if any of the above mentioned restrictions are hit.
     *
     * @param attributeName  the name of the attribute
     * @param attributeValue the value of the attribute
     * @return {@link MacMetadata} 
     */
    @Nonnull
    public MacMetadata addAttribute(@Nonnull String attributeName, @Nonnull String attributeValue) {
        if (metadata.size() == TargetConstants.MAX_METADATA_ITEMS) {
            throw new RuntimeException(
                    "The maximum number of Marketing Cloud Metadata items has been exceeded (" + TargetConstants.MAX_METADATA_ITEMS + ")");
        }

        if (attributeName.length() > TargetConstants.MAX_METADATA_ITEM_LENGTH) {
            throw new RuntimeException(
                    "The attribute name or value exceeds the maximum length of " + TargetConstants.MAX_METADATA_ITEM_LENGTH);
        }

        metadata.put(attributeName, attributeValue);
        return this;
    }

    /**
     * Adds a Marketing Cloud Metadata attribute / value pair to this offer request. The following restrictions apply:
     * <ul>
     * <li>the attributeName or attributeValue must not exceed 250 chars in length</li>
     * <li>The maximum size of the MAC Metadata list is 15 items</li>
     * </ul>
     * This method throws a {@link RuntimeException} if any of the above mentioned restrictions are hit.
     *
     * @param attributeName  the name of the attribute
     * @param attributeValues the values of the attributes
     * @return {@link MacMetadata} 
     */
    public MacMetadata addAttribute(@Nonnull String attributeName, @Nonnull String[] attributeValues) {
        if (metadata.size() == TargetConstants.MAX_METADATA_ITEMS) {
            throw new RuntimeException(
                    "The maximum number of Marketing Cloud Metadata items has been exceeded (" + TargetConstants.MAX_METADATA_ITEMS + ")");
        }

        // the restriction applies to the value in the resulting JSON, so we'll just make some assumptions
        // about the maximum length of the serialized value array
        String value  = StringUtils.join(attributeValues, ",");
        if (attributeName.length() > TargetConstants.MAX_METADATA_ITEM_LENGTH) {
            throw new RuntimeException(
                    "The attribute name or value exceeds the maximum length of " + TargetConstants.MAX_METADATA_ITEM_LENGTH);
        }
        metadata.put(attributeName, attributeValues);
        return this;
    }

    /**
     * Removes a Marketing Cloud Metadata attribute from the list.
     *
     * @param attributeName the name of the attribute.
     */
    public void removeAttribute(@Nonnull String attributeName) {
        metadata.remove(attributeName);
    }

    /**
     * Retrieves the list of the Marketing Cloud Metadata attributes.
     *
     * @return a {@link Map} having the attribute names as keys and the attribute values as values.
     */
    public Map<String, String> getMarketingCloudMetadata() {
        Map<String, String> internalMap = new HashMap<String, String>();

        for(Map.Entry<String, Object> entry: metadata.entrySet()) {
            Object value = entry.getValue();
            if (value instanceof String[]) {
                String[] values = (String[]) value;
                internalMap.put(entry.getKey(), StringUtils.join(values,","));
            } else {
                internalMap.put(entry.getKey(), (String)value);
            }
        }

        return internalMap;
    }


    /**
     * Retrieves the value of a specific attribute from this metadata set
     * @param attributeName the name of the attribute
     * @return a {@link String} containing the value, or {@code null} if the value is not found or it's not a {@link String}.
     */
    @Nullable
    public String getValue(@Nonnull String attributeName) {
        if (StringUtils.isEmpty(attributeName)) {
            return null;
        }

        Object value = metadata.get(attributeName);
        if (value instanceof String[]) {
            return null;
        }

        return (String)metadata.get(attributeName);
    }

    /**
     * Retrieves the value of a specific attribute from this metadata set, specifying what type should the returned value be
     * @param attributeName the name of the attribute
     * @return the value cast to the specified type
     */
    @Nullable
    public String[] getValues(@Nonnull String attributeName) {
        if (StringUtils.isEmpty(attributeName)) {
            return null;
        }

        Object value = metadata.get(attributeName);
        if (value instanceof String[]) {
            return (String[])value;
        } else {
            return new String[] { (String) value};
        }

    }

    /**
     * Generates the mandatory metadata. The mandatory metadata contains the following attributes:
     * <ul>
     * <li>The edit url of the object, i.e. the full URL used to open the object in it's editor in AEM</li>
     * <li>The user that perform the last modification</li>
     * <li>The source product name (in our case Adobe Experience Manager)</li>
     * </ul>
     *
     * @param editUrl          the URL which is used to access the object
     * @param remoteModifiedBy the user which performed the last modification
     * @return a {@link MacMetadata} object containing the three attributes above
     */
    @Nonnull
    public static MacMetadata generateMandatoryMetadata(@Nonnull String editUrl, @Nonnull String remoteModifiedBy) {
        return new MacMetadata().addAttribute(TargetConstants.ATTR_MAC_METADATA_EDIT_URL, editUrl).addAttribute
                (TargetConstants.ATTR_MAC_MEDATATA_SOURCE_PRODUCT, TargetConstants.MAC_METADATA_SOURCE_PRODUCT).addAttribute
                (TargetConstants.ATTR_MAC_METADATA_MODIFIED_BY, remoteModifiedBy);
    }
}
