/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * ___________________
 *
 *  Copyright 2016 Adobe Systems Incorporated
 *  All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Systems Incorporated and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Systems Incorporated.
 **************************************************************************/
package com.day.cq.audit;

import java.util.Arrays;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;

import org.apache.sling.commons.osgi.PropertiesUtil;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public abstract class AbstractAuditLogPurgeRule implements AuditLogPurgeRule {

    // the typical logger

    protected static final Logger LOGGER = LoggerFactory.getLogger(AbstractAuditLogPurgeRule.class);

    private final String auditPath;

    // OSGi configuration params

    private String ruleName;

    private Set<String> eventTypes;

    private String contentPath;

    private int minimumAge;

    public AbstractAuditLogPurgeRule(String auditPath) {
        this.auditPath = auditPath;
    }

    // interface methods

    @Override
    public final String getName() {
        return ruleName;
    }

    @Override
    public final Set<String> getEventTypes() {
        return eventTypes;
    }

    @Override
    public final String getContentPath() {
        return contentPath;
    }

    @Override
    public final int getMinimumAge() {
        return minimumAge;
    }

    // utility methods for activation/deactivation

    protected final void setUp(Map<String, Object> config) {
        ruleName = PropertiesUtil.toString(config.get(NAME), getClass().getName() + System.currentTimeMillis());

        String pathToPurge = PropertiesUtil.toString(config.get(CONTENT_PATH), null);
        if (pathToPurge != null) {
            if ('/' != pathToPurge.charAt(0)) {
                pathToPurge = '/' + pathToPurge;
            }
            contentPath = auditPath + pathToPurge;
        } else {
            contentPath = null;
        }

        String[] type = PropertiesUtil.toStringArray(config.get(TYPES), null);
        if (type != null && type.length > 0) {
            this.eventTypes = new HashSet<String>(Arrays.asList(type));
        } else {
            type = null;
        }

        minimumAge = PropertiesUtil.toInteger(config.get(MINIMUM_AGE), AUDIT_DEFAULT_DAYS);

        LOGGER.info("{} activated", this);
    }

    protected final void tearDown() {
        LOGGER.info("Deactivating {}...", this);
        ruleName = null;
        contentPath = null;
        eventTypes = null;
        minimumAge = 0;
        LOGGER.info("Deactivated");
    }

    @Override
    public final boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null || !(obj instanceof AuditLogPurgeRule)) {
            return false;
        }

        AuditLogPurgeRule other = (AuditLogPurgeRule) obj;

        if (contentPath == null) {
            if (other.getContentPath() != null) {
                return false;
            }
        } else if (!contentPath.equals(other.getContentPath())) {
            return false;
        }

        // if they are both not null
        if (eventTypes == null) {
            if (other.getEventTypes() != null) {
                return false;
            }
        } else if (eventTypes.isEmpty()) { // if they are not both empty
            if (!other.getEventTypes().isEmpty()) {
                return false;
            }
        } else { // check if all elements in the smaller set are included in the bigger one
            Set<String> biggerSet;
            Set<String> smallerSet;

            if (eventTypes.size() >= other.getEventTypes().size()) {
                biggerSet = eventTypes;
                smallerSet = other.getEventTypes();
            } else {
                biggerSet = other.getEventTypes();
                smallerSet = eventTypes;
            }

            for (String current : smallerSet) {
                if (!biggerSet.contains(current)) {
                    return false;
                }
            }
        }

        // ruleName and minimumAge don't really matter if a purge happens on the same path

        return true;
    }

    @Override
    public int hashCode() {
        final int prime = 31;
        int hashCode = 1;

        hashCode = hashCode(prime, hashCode, contentPath);

        if (eventTypes == null || eventTypes.isEmpty()) {
            hashCode = prime * hashCode;
        } else {
            for (String current : eventTypes) {
                hashCode = hashCode(prime, hashCode, current);
            }
        }

        return hashCode;
    }

    private static final <T> int hashCode(final int prime, final int hashCode, T object) {
        return prime * hashCode + ((object == null) ? 0 : object.hashCode());
    }

    @Override
    public final String toString() {
        return "{name = "
               + ruleName
               + ", eventTypes = "
               + eventTypes
               + ", contentPath = "
               + contentPath
               + ", minimumAge = "
               + minimumAge
               + "}";
    }

}
