/*
 * Copyright 1997-2008 Day Management AG
 * Barfuesserplatz 6, 4001 Basel, Switzerland
 * All Rights Reserved.
 *
 * This software is the confidential and proprietary information of
 * Day Management AG, ("Confidential Information"). You shall not
 * disclose such Confidential Information and shall use it only in
 * accordance with the terms of the license agreement you entered into
 * with Day.
 */
package com.day.cq.commons.feed;

import org.apache.sling.api.SlingHttpServletRequest;
import org.apache.sling.api.SlingHttpServletResponse;
import org.apache.sling.api.resource.Resource;
import javax.jcr.RepositoryException;
import java.io.IOException;
import java.util.Calendar;
import java.text.SimpleDateFormat;

/**
 * The <code>RssFeed</code> class prints a resource as an RSS feed.
 */
public class RssFeed extends AbstractFeed {

    /**
     * Creates a new <code>RssFeed</code> instance using the specified
     * servlet request and response.
     * @param req The servlet request
     * @param resp The servlet response
     * @throws RepositoryException if no node can be found
     */
    public RssFeed(SlingHttpServletRequest req, SlingHttpServletResponse resp)
            throws RepositoryException {
        super(null, req, resp);
    }

    /**
     * Creates a new <code>RssFeed</code> instance using the specified
     * resource, servlet request and response.
     * @param res The resource
     * @param req The servlet request
     * @param resp The servlet response
     * @throws RepositoryException if no node can be found
     */
    public RssFeed(Resource res, SlingHttpServletRequest req, SlingHttpServletResponse resp)
            throws RepositoryException {
        super(res, req, resp);
    }

    /**
     * {@inheritDoc}
     */
    public void printHeader() throws IOException {
        initXml();
        xml.openDocument();
        xml.open("rss")
        .attr("version", "2.0")
        .attr("xmlns:atom", "http://www.w3.org/2005/Atom")
            .open("channel")
            .open("link", getHtmlLink(), false).close()
            .open("atom:link")
                .attr("rel", "self")
                .attr("href", getHtmlLink())
                .close()
            .open("title", getTitle(), false).close()
            .open("description", getDescription(), false).close()
            .open("pubDate", getPublishedDate(), false).close();
        if (!"".equals(getLanguage())) {
            xml.open("language", getLanguage(), false).close();
        }
    }

    /**
     * {@inheritDoc}
     */
    public void printEntry() throws IOException {
        String link = isFile() ? getFileLink() : getHtmlLink();
        String comments = isFile() ? "" : getCommentsLink();
        initXml();
        xml.omitXmlDeclaration(true);
        xml.open("item")
            .open("link", link, false).close()
            .open("comments", comments, false).close()
            .open("guid", getFeedLink(), false).close()
            .open("title", getTitle(), false).close()
            .open("description", getDescription(), true).close()
            .open("pubDate", getPublishedDate(), false).close()
            .open("category", getTags(), false).close();
        if (!"".equals(getAuthorEmail())) {
            xml.open("author", getAuthorEmail(), false).close();
        }
        xml.tidyUp();
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public String getContentType() {
        return "application/rss+xml";
    }

    /**
     * {@inheritDoc}
     */
    @Override
    String getFeedEntryPath(Resource res) {
        return res.getPath() + "." + Feed.SELECTOR_FEEDENTRY + "." +
                Feed.SELECTOR_RSS + Feed.SUFFIX_XML;
    }

    /**
     * {@inheritDoc}
     */
    @Override
    protected String getFeedLink() {
        return getUrlPrefix() + getNodePath() + "." + SELECTOR_FEED + 
                "." + SELECTOR_RSS + SUFFIX_XML;
    }

    /**
     * {@inheritDoc}
     */
    @Override
    protected String formatDate(Calendar calendar) {
        // RFC822
        try {
            return new SimpleDateFormat("EEE, d MMM yyyy HH:mm:ss Z").format(calendar.getTime());
        } catch (Exception e) {
            return "";
        }
    }

}