/*
 * Copyright 1997-2008 Day Management AG
 * Barfuesserplatz 6, 4001 Basel, Switzerland
 * All Rights Reserved.
 *
 * This software is the confidential and proprietary information of
 * Day Management AG, ("Confidential Information"). You shall not
 * disclose such Confidential Information and shall use it only in
 * accordance with the terms of the license agreement you entered into
 * with Day.
 */
package com.day.cq.commons.servlets;

import com.day.cq.commons.feed.AtomFeed;
import com.day.cq.commons.feed.Feed;
import com.day.cq.commons.feed.RssFeed;
import org.apache.felix.scr.annotations.sling.SlingServlet;
import org.apache.sling.api.SlingConstants;
import org.apache.sling.api.SlingHttpServletRequest;
import org.apache.sling.api.SlingHttpServletResponse;
import org.apache.sling.api.servlets.SlingSafeMethodsServlet;

import javax.jcr.RepositoryException;
import javax.servlet.ServletException;
import java.io.IOException;

/**
 * The <code>FeedRendererServlet</code> renders the current resource as
 * an Atom or RSS feed, depending on the value of the second selector.
 * If no second selector is passed, it defaults to Atom.
 */
@SlingServlet(
        extensions = "xml",
        selectors = {Feed.SELECTOR_FEED, Feed.SELECTOR_FEEDENTRY},
        resourceTypes = "sling/servlet/default",
        metatype = false,
        generateService = true
)
public class FeedRendererServlet extends SlingSafeMethodsServlet {

    private static final long serialVersionUID = 8394390726904986208L;

    /**
     * Request attribute to use if whole feed should be rendered
     * regardless of any other request attribute.
     */
    public static final String ATTR_NO_INCLUDE = "noinclude";

    /**
     * {@inheritDoc}
     */
    @Override
    protected void doGet(SlingHttpServletRequest req,
                         SlingHttpServletResponse resp)
            throws ServletException, IOException {

        String[] sels = req.getRequestPathInfo().getSelectors();
        final boolean isEntry = (req.getAttribute(ATTR_NO_INCLUDE) == null &&
                req.getAttribute(SlingConstants.ATTR_REQUEST_SERVLET) != null) ||
                (sels.length > 0 && Feed.SELECTOR_FEEDENTRY.equals(sels[0]));
        int maxCount = 30;

        try {
            Feed feed = getFeed(req, resp);
            resp.setContentType(feed.getContentType());
            resp.setCharacterEncoding(feed.getCharacterEncoding());

            if (isEntry) {
                // print single entry
                feed.printEntry();
            } else {
                // print entire feed with child entries
                feed.printHeader();
                feed.printChildEntries(maxCount);
                feed.printFooter();
            }
        } catch (Exception e) {
            throw new ServletException("Error while rendering resource as feed: " + e.getMessage());
        }
    }

    /**
     * Returns an Atom or RSS feed, depending on the value of the second
     * selector. If no second selector is passed, it defaults to Atom.
     * @param req The servlet request
     * @param resp The servlet response
     * @return The feed
     * @throws RepositoryException If no node can be found
     */
    protected Feed getFeed(SlingHttpServletRequest req,
                           SlingHttpServletResponse resp)
            throws RepositoryException {
        String[] sels = req.getRequestPathInfo().getSelectors();
        if (sels.length > 1 && Feed.SELECTOR_RSS.equals(sels[1])) {
            return new RssFeed(req, resp);
        } else {
            return new AtomFeed(req, resp);
        }
    }

}
