/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * ___________________
 *
 *  Copyright 2011 Adobe Systems Incorporated
 *  All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Systems Incorporated and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Systems Incorporated.
 **************************************************************************/

package com.day.cq.dam.api;

import org.apache.sling.api.resource.ResourceResolver;
import org.osgi.annotation.versioning.ProviderType;

import com.adobe.granite.asset.api.AssetRelation;

import javax.jcr.RepositoryException;

import java.util.Iterator;
import java.util.List;
import java.util.Map;

/**
 * The <code>AssetReferenceResolver</code> provides api to resolve the
 * references of a composite assets (for e.g. InDesign assets) to an existing
 * DAM's asset. Further it provides api for creation/cleanup of references and
 * getting list of contained composite Assets. Implementation of this interface
 * should be exposed as a component. 
 * Default implementation of this interface is
 * available at com.day.cq.dam.core.impl.AssetReferenceResolverImpl
 */
@ProviderType
public interface AssetReferenceResolver {

    /**
     * This method would resolve asset references. Maps all the sub assets to
     * DAM's existing asset
     * 
     * @param asset - asset for which references are to be resolved
     * @return - Iterator for relations created
     */
    Iterator<? extends AssetRelation> resolve(Asset asset);

    /**
     * This method would take the property of an referenced asset through which
     * it can be mapped/associated with DAM's existing asset. It returns path of
     * existing asset in DAM on success, otherwise returns NULL in case of not
     * able to resolve.
     * 
     * @param property referenced asset's property through which it can be
     *            mapped to DAM's existing asset.
     * @param resolver ResourceResolver instance
     * @return Path of the resolved asset on success, Null otherwise.
     */
    String getResolvedPath(String property, ResourceResolver resolver);

    /**
     * This method would take the asset path as input and return the assets
     * being referred by the input asset with actual link nodes.
     * 
     * @param assetPath referenced asset's path
     * @param resolver ResourceResolver instance
     * @return Method returns Map in which key would be the composite asset refers
     *         input referenced asset and value would be the actual link node
     *         created by #AssetReferenceResolver.createReference
     */
    Map<String, String> getReferences(String assetPath, ResourceResolver resolver);

    /**
     * This method would create the references which can be associated with the
     * resolved asset as well can be stored as property/node by Composite Asset
     * 
     * @param assetPath composite asset's path
     * @param doSave Whether the repository changes are saved or not.
     * @param resolvedPath resolved path of the referenced asset in DAM
     * @param data Pass the additional data needed for reference creation
     * @param resolver ResourceResolver instance
     * @return path of the created link on success otherwise NULL
     * @throws RepositoryException thrown if an error occurs while accessing the asset
     */
    String createReference(String assetPath, boolean doSave, String resolvedPath, List<Object> data, ResourceResolver resolver)
            throws RepositoryException;

    /**
     * This method will resolve the missing link. If the links are missing at
     * the time of Composite Asset upload, they can be resolved through following
     * method after wards.
     * 
     * @param linkPath link's node path
     * @param resolvedPath path of an existing asset in the DAM to resolved with
     * @param resolver ResourceResolver instance
     * @return returns true flag if able to resolve the link otherwise false
     * @deprecated since 1.2
     */
    @Deprecated
    boolean resolveLink(String linkPath, String resolvedPath, ResourceResolver resolver);

    /**
     * This method will be called at the time of referenced asset deletion and
     * provides the cleanup of the associated links.
     * 
     * @param linkPath link's node path
     * @param resolver ResourceResolver instance
     * @return returns true flag on successful link deletion otherwise false
     * @deprecated since 1.2
     */
    @Deprecated
    boolean deleteLink(String linkPath, ResourceResolver resolver);

    /**
     * This method will clean all the links node created for referenced asset.
     * It calls #AssetReferenceResolver.deleteLink internally to delete
     * individual links
     * 
     * @param assetPath referenced asset's path
     * @param resolver ResourceResolver instance
     * @return returns true flag on successful cleanup otherwise false
     * @deprecated since 1.2
     */
    @Deprecated
    boolean cleanup(String assetPath, ResourceResolver resolver);

}