/*
 * Copyright 1997-2009 Day Management AG
 * Barfuesserplatz 6, 4001 Basel, Switzerland
 * All Rights Reserved.
 *
 * This software is the confidential and proprietary information of
 * Day Management AG, ("Confidential Information"). You shall not
 * disclose such Confidential Information and shall use it only in
 * accordance with the terms of the license agreement you entered into
 * with Day.
 */
package com.day.cq.dam.commons.handler;

import com.day.cq.dam.commons.handler.FilterStateListener;
import com.day.cq.dam.commons.handler.Filter;

import java.io.IOException;
import java.io.OutputStream;

/**
 * Continuously filter bytes and produce an output that can be read
 * back. Optionally pass processed bytes to a next filter.
 *
 * @author dpfister
 */
public abstract class AbstractFilter implements Filter {

    /**
     * Next filter in chain.
     */
    protected Filter nextFilter;

    /**
     * Stream position.
     */
    protected int streamPos;

    /**
     * Filter state listener.
     */
    protected FilterStateListener listener;

    /**
     * Output stream return by listener.
     */
    protected OutputStream out;

    /**
     * Flag indicating whether the filter automatically resets its state
     * when an end marker has been processed.
     */
    protected boolean autoReset;

    /**
     * Create a new instance of this class.
     *
     * @param nextFilter next filter, may be <code>null</code>
     */
    protected AbstractFilter(AbstractFilter nextFilter) {
        this.nextFilter = nextFilter;
    }

    /**
     * Create a new instance of this class. Empty constructor, with no
     * next filter.
     */
    protected AbstractFilter() {
        this(null);
    }

    /**
     * Reset internal state of this filter. Default implementation clears
     * output stream returned from a listener notification
     */
    protected void doReset() {
        out = null;
    }

    /**
     * Write a couple of bytes to this filter and process them (overridable).
     *
     * @param buf buffer
     * @param off offset inside buffer
     * @param len number of bytes
     * @throws IOException if an error occurs
     */
    protected abstract void doFilter(byte[] buf, int off, int len) throws IOException;

    /**
     * Write some byte to the output stream.
     *
     * @param b byte
     * @throws IOException if an I/O error occurs
     */
    protected void put(int b) throws IOException {
        put(new byte[] { (byte) b });
    }

    /**
     * Write some bytes to the output stream.
     *
     * @param buf buffer
     * @throws IOException if an I/O error occurs
     */
    protected void put(byte[] buf) throws IOException {
        if (nextFilter != null) {
            nextFilter.filter(buf, 0, buf.length);
        }
        if (out != null) {
            out.write(buf, 0, buf.length);
        }
    }

    /**
     * Write some bytes to the output stream.
     *
     * @param buf buffer
     * @param off offset
     * @param len number of bytes
     * @throws IOException if an I/O error occurs
     */
    protected void put(byte[] buf, int off, int len) throws IOException {
        if (nextFilter != null) {
            nextFilter.filter(buf, 0, buf.length);
        }
        if (out != null) {
            out.write(buf, off, len);
        }
    }

    /**
     * Set a filter state listener.
     *
     * @param listener listener
     */
    public void setFilterStateListener(FilterStateListener listener) {
        this.listener = listener;
    }

    /**
     * Invoked when the scanner found a starting marker. Default implementation
     * tells next filter to reset its internal state.
     */
    protected void notifyStarted() {
        if (nextFilter != null) {
            nextFilter.reset();
        }
        if (listener != null) {
            out = listener.started(this);
        }
    }

    /**
     * Invoked when the scanner found an ending marker. Default implementation
     * tells next filter that it should finish processing. If the filter has
     * its <code>autoReset</code> setting enabled, the internal state gets
     * reset.
     */
    protected void notifyEnded() {
        if (listener != null) {
            listener.ended(this, out);
        }
        if (autoReset) {
            reset();
        }
    }

    //------------------------------------------------------------------- Filter

    /**
     * {@inheritDoc}
     */
    public final void filter(byte[] buf, int off, int len) throws IOException {
        doFilter(buf, off, len);

        streamPos += len;
    }

    /**
     * {@inheritDoc}
     */
    public final void reset() {
        if (nextFilter != null) {
            nextFilter.reset();
        }
        doReset();
    }

    /**
     * {@inheritDoc}
     */
    public void setAutoReset(boolean autoReset) {
        this.autoReset = autoReset;
    }

    /**
     * {@inheritDoc}
     *
     * This implementation returns <code>true</code>.
     */
    public boolean isValid() {
        return true;
    }
}
