/******************************************************************************
 * ADOBE CONFIDENTIAL
 * ___________________
 *
 * Copyright 2024 Adobe
 * All Rights Reserved.
 *
 * NOTICE: All information contained herein is, and remains
 * the property of Adobe and its suppliers, if any. The intellectual
 * and technical concepts contained herein are proprietary to Adobe
 * and its suppliers and are protected by all applicable intellectual
 * property laws, including trade secret and copyright laws.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe.
 ******************************************************************************/

package com.day.cq.dam.commons.metadata;

import com.adobe.xmp.schema.service.SchemaServiceException;
import com.adobe.xmp.schema.service.StandardRelaxNGProvider;

import java.io.IOException;
import java.io.InputStream;
import java.net.URI;
import java.net.URISyntaxException;
import java.util.HashSet;
import java.util.Set;

/**
 * Provides a small set of RelaxNG schema resources for the internal dam: XMP namespace, and delegates to the default
 * StandardRelaxNGProvider behavior for all other XMP namespaces.
 * The {@link com.adobe.xmp.schema.service.StandardRelaxNGProvider#StandardRelaxNGProvider()} default constructor
 * parses the RNG schema included in its own bundle (explicitly using StandardRelaxNGProvider.class.getClassLoader())
 * to map resource URIs, so we don't have to worry about OSGi wiring getting in the way of opening InputStreams for those
 * files from this bundle.
 */
class DamRelaxNGProvider extends StandardRelaxNGProvider {
    static final String NS_DAM = "http://www.day.com/dam/1.0";
    static final String DAM_SCHEMA_FULL_NAME = "xmpDamSchema/AdobePacket-dam.rng";
    static final String DAM_DEFS_NAME = "AdobeDefinitions-dam.rng";
    static final String DAM_DEFS_FULL_NAME = "xmpDamSchema/properties/" + DAM_DEFS_NAME;

    DamRelaxNGProvider() throws SchemaServiceException {
        super();
    }

    @Override
    public InputStream getSchema(String namespaceURI) throws IOException {
        if (NS_DAM.equals(namespaceURI)) {
            try {
                return getResource(DAM_SCHEMA_FULL_NAME).toURL().openStream();
            } catch (URISyntaxException e) {
                throw new IOException(e.getMessage());
            }
        }
        return super.getSchema(namespaceURI);
    }

    @Override
    public InputStream getInclude(String includeName) throws IOException {
        if (DAM_DEFS_NAME.equals(includeName)) {
            try {
                return getResource(DAM_DEFS_FULL_NAME).toURL().openStream();
            } catch (URISyntaxException e) {
                throw new IOException(e.getMessage());
            }
        }
        return super.getInclude(includeName);
    }

    @Override
    public boolean isAvailable(String namespaceURI) {
        return NS_DAM.equals(namespaceURI) || super.isAvailable(namespaceURI);
    }

    @Override
    public Set<String> getNamespaces() {
        Set<String> namespaces = new HashSet<>();
        namespaces.add(NS_DAM);
        namespaces.addAll(super.getNamespaces());
        return namespaces;
    }

    private static ClassLoader getClassLoader() {
        ClassLoader loader = DamRelaxNGProvider.class.getClassLoader();
        if (loader == null) {
            loader = ClassLoader.getSystemClassLoader();
        }
        return loader;
    }

    /**
     * @return Returns the absolute path of this class' folder.
     * @throws java.net.URISyntaxException If the base folder cannot be determined
     */
    private static URI getResource(String path) throws URISyntaxException {
        ClassLoader loader = getClassLoader();
        return loader.getResource(path).toURI();
    }
}
