/*************************************************************************
 * ADOBE CONFIDENTIAL __________________
 * <p>
 * Copyright 2016 Adobe Systems Incorporated All Rights Reserved.
 * <p>
 * NOTICE:  All information contained herein is, and remains the property of Adobe Systems Incorporated and its suppliers, if any.  The
 * intellectual and technical concepts contained herein are proprietary to Adobe Systems Incorporated and its suppliers and are protected by
 * trade secret or copyright law. Dissemination of this information or reproduction of this material is strictly forbidden unless prior
 * written permission is obtained from Adobe Systems Incorporated.
 **************************************************************************/
package com.day.cq.dam.commons.metadata;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.Writer;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Dictionary;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.Stack;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import javax.xml.namespace.NamespaceContext;
import javax.xml.namespace.QName;
import javax.xml.stream.Location;
import javax.xml.stream.XMLEventReader;
import javax.xml.stream.XMLEventWriter;
import javax.xml.stream.XMLInputFactory;
import javax.xml.stream.XMLOutputFactory;
import javax.xml.stream.XMLResolver;
import javax.xml.stream.XMLStreamException;
import javax.xml.stream.events.Attribute;
import javax.xml.stream.events.Characters;
import javax.xml.stream.events.EndElement;
import javax.xml.stream.events.StartElement;
import javax.xml.stream.events.XMLEvent;

import org.apache.felix.scr.annotations.Activate;
import org.apache.felix.scr.annotations.Component;
import org.apache.felix.scr.annotations.Property;
import org.apache.felix.scr.annotations.Service;
import org.apache.sling.commons.osgi.PropertiesUtil;
import org.osgi.service.component.ComponentContext;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * The <code>XmpFilter</code> class is used to reduce Extensible Metadata Platform (XMP) properties to a desired subset using various
 * configuration options such as white- and blacklists.
 */
@Component(metatype = true, label = "Adobe CQ DAM XmpFilter",
        description = "Filtering Xmp Properties by block-/allow-listing names and namespaces")
@Service
public class XmpFilterBlackWhite implements XmpFilter {

    private static final String RDF_NS = "http://www.w3.org/1999/02/22-rdf-syntax-ns#";

    private static final String LN_ALT = "Alt";
    private static final String LN_BAG = "Bag";
    private static final String LN_DESCRIPTION = "Description";
    private static final String LN_LI = "li";
    private static final String LN_RDF = "RDF";
    private static final String LN_SEQ = "Seq";

    private static final QName N_ALT = new QName(RDF_NS, LN_ALT);
    private static final QName N_BAG = new QName(RDF_NS, LN_BAG);
    private static final QName N_DESCRIPTION = new QName(RDF_NS, LN_DESCRIPTION);
    private static final QName N_LI = new QName(RDF_NS, LN_LI);
    private static final QName N_RDF = new QName(RDF_NS, LN_RDF);
    private static final QName N_SEQ = new QName(RDF_NS, LN_SEQ);

    private static final Set<QName> RdfContainerNames;
    private static final Set<QName> IgnoredPropertyAttributeNames;
    private static final Set<String> IgnoredPropertyAttributeNamespaces;
    
    private static final boolean DEFAULT_APPLY_ALLOW_LIST = false;
    private static final boolean DEFAULT_APPLY_BLOCKLIST = true;

    static {
        Set<QName> names = new HashSet<QName>();
        names.add(N_ALT);
        names.add(N_BAG);
        names.add(N_SEQ);
        RdfContainerNames = Collections.unmodifiableSet(names);
        names = new HashSet<QName>();
        IgnoredPropertyAttributeNames = Collections.unmodifiableSet(names);
        Set<String> namespaces = new HashSet<String>();
        namespaces.add(RDF_NS);
        IgnoredPropertyAttributeNamespaces = Collections.unmodifiableSet(namespaces);
    }

    private static final XMLResolver DEFAULT_XML_RESOLVER = new XMLResolver() {
        @Override
        public Object resolveEntity(String publicID, String systemID, String baseURI, String namespace) throws XMLStreamException {
            log.debug("Resolution of external entities in XML payload not supported - publicId: " + publicID + ", systemId: " + systemID);
            throw new XMLStreamException("This parser does not support resolution of external entities (publicId: " + publicID
                    + ", systemId: " + systemID + ")");
        }
    };

    @Property(boolValue = DEFAULT_APPLY_ALLOW_LIST,
            label = "Apply AllowList to XMP Properties",
            description = "Only let the allowed xmp properties through, applied before any blocklist")
    public static final String APPLY_ALLOWLIST = "xmp.filter.apply_allowlist";
    public static final String ApplyWhiteList = "xmp.filter.apply_whitelist";
    
    @org.apache.felix.scr.annotations.Property(value = "",
            label = "Allowed XML Names for XMP filtering",
            description = "XML Names, such as '{namespace-uri}name', '{namespace-uri}*', 'prefix:name' and 'prefix:*', passed on during "
                    + "XMP filtering. Use '[>n]' or '[<n]' to limit the values accepted for multi-valued XMP properties. Example: "
                    + "'history[<100]' will only let the first 99 values through", cardinality = Integer.MAX_VALUE)
    public static final String ALLOWLIST = "xmp.filter.allowlist";
    public static final String WhiteList = "xmp.filter.whitelist";
    
    @Property(boolValue = DEFAULT_APPLY_BLOCKLIST,
            label = "Apply Blocklist to XMP Properties",
            description = "Filter out the blocked xmp properties, applied after any allowlisting")
    public static final String APPLY_BLOCKLIST = "xmp.filter.apply_blocklist";
    public static final String ApplyBlackList = "xmp.filter.apply_blacklist";
    

    /**
     * CQ-4326783: two more default xmpfilter (xmpMM:Pantry and xmpMM:History) were added to avoid the following two
     * issues
     * 1. a slow UI performance 
     * 2. a customer's environment crashed during metadata extraction
     */
    @Property(value = {
            "{http://ns.adobe.com/photoshop/1.0/}DocumentAncestors[>100]",
            "xmpMM:Pantry",
            "xmpMM:History"
    },
            label = "Blocklisted XML Names for XMP filtering",
            description = "XML Names, such as '{namespace-uri}name', '{namespace-uri}*', 'prefix:name' and 'prefix:*', filtered out during "
                    + "XMP processing. Use '[>n]' or '[<n]' to limit the values removed for multi-valued XMP properties. Example: "
                    + "'history[>100]' discards the 101st and following values.", cardinality = Integer.MAX_VALUE)
    public static final String BLOCKLIST = "xmp.filter.blocklist";
    public static final String BlackList = "xmp.filter.blacklist";
    

    private static final Logger log = LoggerFactory.getLogger(XmpFilterBlackWhite.class);

    private static class NameDef {

        private static final Pattern RE_LIMIT_MAX = Pattern.compile("(.*)\\[([><]?)(\\d+)\\]");

        String orig;
        String uri;
        String localName;
        String prefix;
        EventCondition condition;
        Object condToken;

        void init(String s) {
            orig = s;
            uri = null;
            localName = null;
            prefix = null;
            condition = null;
            condToken = null;

            if (s.charAt(0) == '{') {
                int idx = s.lastIndexOf('}');
                if (idx <= 0) {
                    throw new IllegalArgumentException("'}' missing in allowlist name: " + s);
                }
                uri = s.substring(1, idx);
                localName = (idx < s.length() - 1) ? s.substring(idx + 1) : "*";
            } else {
                int idx = s.indexOf(':');
                prefix = (idx > 0) ? s.substring(0, idx) : "";
                localName = (idx >= 0) ? ((idx < s.length() - 1) ? s.substring(idx + 1) : "*") : s;
                if (localName.length() == 0) {
                    throw new IllegalArgumentException("'}' empty allowlist name: " + s);
                }
            }
            Matcher m = RE_LIMIT_MAX.matcher(localName);
            if (m.matches()) {
                try {
                    String op = m.group(2);
                    int n = Integer.parseInt(m.group(3));
                    localName = m.group(1);
                    if ("".equals(localName)) {
                        localName = "*";
                    }
                    condition = "<".equals(op) ? new MinIndexCondition(n) : new MaxIndexCondition(n);
                    condToken = (uri != null) ? ("*".equals(localName) ? uri : new QName(uri, localName)) : prefix + ":" + localName;
                } catch (NumberFormatException ex) {
                    log.debug("error pasing max confition from {}", orig, ex);
                }
            }
            /*
             * One last special: if localName == '*' now, and prefix == '' and uri == null and the original string had no ':',
             * we had a '*' or '*[condition]' specification. Treat that equivalent to '{*}*' resp. '{*}*[condition]'
             */
            if ("*".equals(localName) && "".equals(prefix) && uri == null && orig.indexOf(':') < 0) {
                prefix = null;
                uri = "*";
                if (condition != null) {
                    condToken = uri;
                }
            }
        }
    }

    private static NameConditionProvider parseNameList(Object list, boolean exclude, XmpFilterMode mode) {
        Set<String> namespaces = new HashSet<String>();
        Map<String, Set<String>> names = new HashMap<String, Set<String>>();
        Set<String> prefixes = new HashSet<String>();
        Map<String, Set<String>> prefixedNames = new HashMap<String, Set<String>>();
        Map<Object, EventCondition> conditions = new HashMap<Object, EventCondition>();
        NameDef n = new NameDef();
        for (String s : PropertiesUtil.toStringArray(list, new String[]{})) {
            if (s.length() <= 0) continue;
            n.init(s);
            if (n.uri != null) {
                if ("*".equals(n.localName)) {
                    namespaces.add(n.uri);
                } else {
                    if (!names.containsKey(n.uri)) {
                        names.put(n.uri, new HashSet<String>());
                    }
                    names.get(n.uri).add(n.localName);
                }
            } else {
                if ("*".equals(n.localName)) {
                    prefixes.add(n.prefix);
                } else {
                    if (!prefixedNames.containsKey(n.prefix)) {
                        prefixedNames.put(n.prefix, new HashSet<String>());
                    }
                    prefixedNames.get(n.prefix).add(n.localName);
                }
            }
            if (n.condition != null) {
                conditions.put(n.condToken, n.condition);
            }
        }
        return new NameConditionProvider(namespaces, names, prefixes, prefixedNames, conditions, exclude, mode);
    }

    private enum XmpFilterMode {
        FILTER,
        SIEVE
    }

    interface EventCondition {
        EventCondition ACCEPT = new AcceptCondition();
        EventCondition DENY = new DenyCondition();

        boolean acceptsValue(String value);

        boolean accepts(XMLEvent event, List<XMLEvent> queue);

        EventCondition newInstance(boolean exclude, XmpFilterMode mode);
    }

    private static class AcceptCondition implements EventCondition {
        @Override
        public boolean accepts(XMLEvent event, List<XMLEvent> queue) {
            queue.add(event);
            return true;
        }

        @Override
        public boolean acceptsValue(String value) {
            return true;
        }

        @Override
        public EventCondition newInstance(boolean exclude, XmpFilterMode mode) {
            return exclude ? DENY : ACCEPT;
        }

    }

    private static class DenyCondition implements EventCondition {
        @Override
        public boolean accepts(XMLEvent event, List<XMLEvent> queue) {
            return false;
        }

        @Override
        public boolean acceptsValue(String value) {
            return false;
        }

        @Override
        public EventCondition newInstance(boolean exclude, XmpFilterMode mode) {
            return exclude ? ACCEPT : DENY;
        }
    }

    private static class AndCondition implements EventCondition {

        private final EventCondition cond1;
        private final EventCondition cond2;

        AndCondition(EventCondition cond1, EventCondition cond2) {
            this.cond1 = cond1;
            this.cond2 = cond2;
        }

        @Override
        public boolean acceptsValue(String value) {
            return cond1.acceptsValue(value) && cond2.acceptsValue(value);
        }

        @Override
        public boolean accepts(XMLEvent event, List<XMLEvent> queue) {
            int n = queue.size();
            if (!cond1.accepts(event, queue)) {
                return false;
            }
            /* Need to avoid duplicate event additions to queue */
            while (queue.size() > n) {
                queue.remove(queue.size() - 1);
            }
            return cond2.accepts(event, queue);
        }

        @Override
        public EventCondition newInstance(boolean inverse, XmpFilterMode mode) {
            return inverse ? new OrCondition(cond1.newInstance(true, mode), cond2.newInstance(true, mode))
                    : new AndCondition(cond1.newInstance(false, mode), cond2.newInstance(false, mode));
        }
    }

    private static class OrCondition implements EventCondition {

        private final EventCondition cond1;
        private final EventCondition cond2;

        OrCondition(EventCondition cond1, EventCondition cond2) {
            this.cond1 = cond1;
            this.cond2 = cond2;
        }

        @Override
        public boolean acceptsValue(String value) {
            return cond1.acceptsValue(value) || cond2.acceptsValue(value);
        }

        @Override
        public boolean accepts(XMLEvent event, List<XMLEvent> queue) {
            return (cond1.accepts(event, queue) || cond2.accepts(event, queue));
        }

        @Override
        public EventCondition newInstance(boolean exclude, XmpFilterMode mode) {
            return exclude ? new AndCondition(cond1.newInstance(true, mode), cond2.newInstance(true, mode))
                    : new OrCondition(cond1.newInstance(false, mode), cond2.newInstance(false, mode));
        }
    }

    private static abstract class AbstractIndexCondition implements EventCondition {
        private final int index;
        private int level;
        private int itemCount = 0;
        // indicates if further events should be ignored
        private boolean ignore = false;
        private boolean insideContainer = false;

        AbstractIndexCondition(int index) {
            this.index = index;
        }

        protected int getIndex() {
            return index;
        }

        protected int getItemCount() {
            return itemCount;
        }

        protected abstract boolean shouldAccept();

        @Override
        public boolean acceptsValue(String value) {
            return shouldAccept();
        }

        @Override
        public boolean accepts(XMLEvent event, List<XMLEvent> queue) {
            QName name;
            boolean accept = true;

            /**
             * Following FSM designed for structure like:
             * <ps:Ancestors> level = 1, condition starts here [accept = true]
             *     <rdf:Seq>  level = 2, container level       [accept = true]
             *         <li abc="xyz">   level = 3, item level            [accept = false if index condition is not
             *         accepted and ignore further levels]
             *             <abc>  level = 4, inside item       [accept = false if parent item wasn't accepted
             *                                                  (ignore == true)]
             *             </abc>
             *         </li>
             *     </rdf:Seq>
             * </ps:Ancestors>
             *
             * ERROR: Bad condition, if the condition is applied to structure different then above.
             */
            if (event.getEventType() == XMLEvent.START_ELEMENT) {
                name = event.asStartElement().getName();
                level++;

                if (level == 1) {
                    accept = true;
                } else if (level == 2) {
                    if (!RdfContainerNames.contains(name)) {
                        log.warn("Bad condition, expected XMP container but found: {}", name);
                        insideContainer = false;
                    } else {
                        insideContainer = true;
                    }
                    accept = true;
                } else if (level == 3) {
                    //evaluate the index condition only at item level and inside array containers.
                    accept = true;
                    if (insideContainer && N_LI.equals(name)) {
                        // increase item count
                        itemCount++;
                        if (!shouldAccept()) {
                            accept = false;
                            //start ignoring further events until this item ends.
                            ignore = true;
                        }
                    } else if (insideContainer) {
                        log.warn("Bad XMP, expected Array Element but found: {}", name);
                    } // else {} `WARN` already logged in at `level == 2`
                }

                if (level > 3 ) {
                    if (ignore) {
                        accept = false;
                    } else {
                        accept = true;
                    }
                }
            } else if (event.getEventType() == XMLEvent.END_ELEMENT) {
                name = event.asEndElement().getName();
                if (level == 1) {
                    accept = true;
                } else if (level == 2) {
                    accept = true;
                } else if (level == 3) {
                    if (ignore) {
                        accept = false;
                    } else {
                        accept = true;
                    }
                    // item ends here, stop ignoring.
                    ignore = false;
                }

                if (level > 3) {
                    if (ignore) {
                        accept = false;
                    } else {
                        accept = true;
                    }
                }

                level--;
            } else if (ignore) {
                accept = false;
            }

            if (accept) {
                queue.add(event);
            }
            return accept;
        }
    }

    private static class MaxIndexCondition extends AbstractIndexCondition {

        MaxIndexCondition(int maxIndex) {
            super(maxIndex);
        }

        protected boolean shouldAccept() {
            return (getItemCount() <= getIndex());
        }

        @Override
        public EventCondition newInstance(boolean inverse, XmpFilterMode mode) {
            if (mode == XmpFilterMode.SIEVE) {
                return ACCEPT.newInstance(inverse, mode);
            }
            return inverse ? new MaxIndexCondition(getIndex()) : new MinIndexCondition(getIndex() + 1);
        }
    }

    private static class MinIndexCondition extends AbstractIndexCondition {

        MinIndexCondition(int minIndex) {
            super(minIndex);
        }

        protected boolean shouldAccept() {
            return (getItemCount() >= getIndex());
        }

        @Override
        public EventCondition newInstance(boolean inverse, XmpFilterMode mode) {
            if (mode == XmpFilterMode.SIEVE) {
                return ACCEPT.newInstance(inverse, mode);
            }
            return inverse ? new MinIndexCondition(getIndex()) : new MaxIndexCondition(getIndex() - 1);
        }
    }

    interface EventConditionProvider {

        /**
         * Given an XMP property name, determine the {@link EventCondition} instance that applies.
         *
         * @param name name of the XMP property to retrieve condition for
         * @return the condition that applies
         */
        EventCondition getCondition(QName name);

        boolean isEmpty();

        EventConditionProvider newInstance(XmpFilterMode mode);
    }

    private static class NameConditionProvider implements EventConditionProvider {

        private final Set<String> namespaces;
        private final Map<String, Set<String>> names;
        private final Set<String> prefixes;
        private final Map<String, Set<String>> prefixedNames;
        private final Map<Object, EventCondition> conditions;
        private final boolean exclude;
        private final XmpFilterMode mode;
        Map<Object, EventCondition> env;

        NameConditionProvider(Set<String> namespaces, Map<String, Set<String>> names,
                              Set<String> prefixes, Map<String, Set<String>> prefixedNames,
                              Map<Object, EventCondition> conditions, boolean exclude,
                              XmpFilterMode mode) {
            this.namespaces = namespaces;
            this.names = names;
            this.prefixes = prefixes;
            this.prefixedNames = prefixedNames;
            this.conditions = conditions;
            this.exclude = exclude;
            this.mode = mode;
            this.env = Collections.emptyMap();
        }

        NameConditionProvider(NameConditionProvider base, boolean exclude, XmpFilterMode mode) {
            this.namespaces = base.namespaces;
            this.names = base.names;
            this.prefixes = base.prefixes;
            this.prefixedNames = base.prefixedNames;
            this.conditions = base.conditions;
            this.exclude = exclude;
            this.mode = mode;
            this.env = new HashMap<Object, EventCondition>();
        }

        /**
         * Given an XMP property name, determine the {@link EventCondition} instance that applies.
         *
         * @param name name of the XMP property to retrieve condition for
         * @return the condition that applies
         */
        public EventCondition getCondition(QName name) {
            Object token = getConditionToken(name);
            if (token == null) {
                return exclude ? EventCondition.ACCEPT : EventCondition.DENY;
            }
            EventCondition cond = env.get(token);
            if (cond == null) {
                cond = getDefinedCondition(token, name).newInstance(exclude, this.mode);
                env.put(token, cond);
            }
            return cond;
        }

        public boolean isEmpty() {
            return conditions.isEmpty() && namespaces.isEmpty() && names.isEmpty()
                    && prefixes.isEmpty() && prefixedNames.isEmpty();
        }

        public EventConditionProvider newInstance(XmpFilterMode mode) {
            if (this.mode == mode) {
                return new NameConditionProvider(this, exclude, mode);
            }
            return new NameConditionProvider(this, !exclude, mode);
        }

        private Object getConditionToken(QName qname) {
            final String uri = qname.getNamespaceURI();
            final String localName = qname.getLocalPart();
            if (namespaces.contains(uri) || (names.containsKey(uri) && names.get(uri).contains(localName))) {
                return qname;
            }
            final String prefix = qname.getPrefix();
            if (prefixes.contains(prefix) || (prefixedNames.containsKey(prefix)
                    && prefixedNames.get(prefix).contains(localName))) {
                return prefix + ":" + localName;
            }
            /* Check for wildcards */
            if (namespaces.contains("*") || names.containsKey("*") && names.get("*").contains(localName)) {
                return new QName("*", localName);
            }
            if (prefixes.contains("*") || (prefixedNames.containsKey("*") && prefixedNames.get("*").contains(localName))) {
                return "*:" + localName;
            }
            return null;
        }

        private EventCondition getDefinedCondition(Object token, QName name) {
            EventCondition cond = conditions.get(token);
            if (cond == null) {
                cond = conditions.get(name.getNamespaceURI());
            }
            if (cond == null) {
                cond = conditions.get(name.getPrefix() + ":*");
            }
            if (cond == null) {
                cond = conditions.get(new QName("*", name.getLocalPart()));
            }
            if (cond == null) {
                cond = conditions.get("*:" + name.getLocalPart());
            }
            if (cond == null) {
                cond = conditions.get("*:*");
            }
            if (cond == null) {
                cond = conditions.get("*");
            }
            if (cond == null) {
                cond = EventCondition.ACCEPT;
            }
            return cond;
        }

    }

    private static class AndConditionProvider implements EventConditionProvider {

        final EventConditionProvider cond1;
        final EventConditionProvider cond2;
        final Map<QName, EventCondition> env;

        AndConditionProvider(EventConditionProvider white, EventConditionProvider black, XmpFilterMode mode) {
            this.cond1 = white.newInstance(mode);
            this.cond2 = black.newInstance(mode);
            this.env = new HashMap<QName, EventCondition>();
        }

        @Override
        public EventCondition getCondition(QName name) {
            EventCondition cond = env.get(name);
            if (cond == null) {
                EventCondition cw = cond1.getCondition(name);
                if (cw == EventCondition.ACCEPT) {
                    return cond2.getCondition(name);
                }
                EventCondition cb = cond2.getCondition(name);
                if (cb == EventCondition.ACCEPT) {
                    return cw;
                }
                cond = new AndCondition(cw, cb);
                env.put(name, cond);
            }
            return cond;
        }

        @Override
        public boolean isEmpty() {
            return cond1.isEmpty() && cond2.isEmpty();
        }

        public EventConditionProvider newInstance(XmpFilterMode mode) {
            return new AndConditionProvider(cond1, cond2, mode);
        }
    }

    private static class OrConditionProvider implements EventConditionProvider {

        final EventConditionProvider cond1;
        final EventConditionProvider cond2;
        final Map<QName, EventCondition> env;

        OrConditionProvider(EventConditionProvider white, EventConditionProvider black, XmpFilterMode mode) {
            this.cond1 = white.newInstance(mode);
            this.cond2 = black.newInstance(mode);
            this.env = new HashMap<QName, EventCondition>();
        }

        @Override
        public EventCondition getCondition(QName name) {
            EventCondition cond = env.get(name);
            if (cond == null) {
                EventCondition cw = cond1.getCondition(name);
                if (cw == EventCondition.DENY) {
                    return cond2.getCondition(name);
                }
                EventCondition cb = cond2.getCondition(name);
                if (cb == EventCondition.DENY) {
                    return cw;
                }
                cond = new OrCondition(cw, cb);
                env.put(name, cond);
            }
            return cond;
        }

        @Override
        public boolean isEmpty() {
            return cond1.isEmpty() && cond2.isEmpty();
        }

        public EventConditionProvider newInstance(XmpFilterMode mode) {
            return new OrConditionProvider(cond1, cond2, mode);
        }
    }

    private EventConditionProvider filterConditions;
    private EventConditionProvider sieveConditions;

    public XmpFilterBlackWhite() {
        log.trace("instantiated");
    }

    @SuppressWarnings("rawtypes")
    public void setConfig(Dictionary cfg) {
        NameConditionProvider allow = null;
        NameConditionProvider block = null;
        NameConditionProvider oldAllow = parseNameList(cfg.get(WhiteList), false, XmpFilterMode.FILTER);
        NameConditionProvider oldBlock = parseNameList(cfg.get(BlackList), true, XmpFilterMode.FILTER);
        boolean applyAllowlist = false;
        boolean applyBlocklist = true;
        
        if (parseNameList(cfg.get(ALLOWLIST), false, XmpFilterMode.FILTER) != null)
         allow = parseNameList(cfg.get(ALLOWLIST), false, XmpFilterMode.FILTER);
        else if (oldAllow != null)
            allow = oldAllow;
        
        if (parseNameList(cfg.get(BLOCKLIST), true, XmpFilterMode.FILTER) != null)
         block = parseNameList(cfg.get(BLOCKLIST), true, XmpFilterMode.FILTER);
        else if (oldBlock != null)
            block = oldBlock;
        //If new non-default values are present use them else check for old values.
        if(PropertiesUtil.toBoolean(cfg.get(APPLY_ALLOWLIST), DEFAULT_APPLY_ALLOW_LIST) != DEFAULT_APPLY_ALLOW_LIST)
        	applyAllowlist = PropertiesUtil.toBoolean(cfg.get(APPLY_ALLOWLIST), DEFAULT_APPLY_ALLOW_LIST);
        else if (PropertiesUtil.toBoolean(cfg.get(ApplyWhiteList), DEFAULT_APPLY_ALLOW_LIST) != DEFAULT_APPLY_ALLOW_LIST)
        	applyAllowlist = PropertiesUtil.toBoolean(cfg.get(ApplyWhiteList), DEFAULT_APPLY_ALLOW_LIST);
        
        if(PropertiesUtil.toBoolean(cfg.get(APPLY_BLOCKLIST), DEFAULT_APPLY_BLOCKLIST) != DEFAULT_APPLY_BLOCKLIST && !block.isEmpty())
        	applyBlocklist = PropertiesUtil.toBoolean(cfg.get(APPLY_BLOCKLIST), DEFAULT_APPLY_BLOCKLIST) ;
        else if (PropertiesUtil.toBoolean(cfg.get(ApplyBlackList), DEFAULT_APPLY_BLOCKLIST) != DEFAULT_APPLY_BLOCKLIST && !oldBlock.isEmpty())
        	applyBlocklist = PropertiesUtil.toBoolean(cfg.get(ApplyBlackList), DEFAULT_APPLY_BLOCKLIST);
        
        if (applyAllowlist && applyBlocklist) {
            this.filterConditions = new AndConditionProvider(allow, block, XmpFilterMode.FILTER);
            this.sieveConditions = new OrConditionProvider(block.newInstance(XmpFilterMode.SIEVE),
                    allow.newInstance(XmpFilterMode.SIEVE), XmpFilterMode.SIEVE);
        } else if (applyAllowlist) {
            this.filterConditions = allow;
            this.sieveConditions = allow.newInstance(XmpFilterMode.SIEVE);
        } else if (applyBlocklist) {
            this.filterConditions = block;
            this.sieveConditions = block.newInstance(XmpFilterMode.SIEVE);
        } else {
            this.filterConditions = null; /* accept all */
            this.sieveConditions = null;
        }
    }

    @Activate
    protected void activate(ComponentContext ctx) {
        /* Parse our configuration parameters into the proper sets, so that we have fast lookup later on.
         */
        log.trace("activate");
        setConfig(ctx.getProperties());
    }

    @Override
    public boolean isActive() {
        return this.filterConditions != null;
    }

    @Override
    public InputStream filter(InputStream xmpIS) throws IOException {
        log.debug("filter");
        if (!isActive()) {
            /* no filtering happening */
            return xmpIS;
        }
        /* Filter: in(whitelist) && !in(blacklist) */
        XMLInputFactory xmlif = getInputFactory();
        try {
            XMLEventFilter filter = new XMLReaderEventFilter(xmlif.createXMLEventReader(xmpIS));
            if (filterConditions != null) {
                filter = new XmpPropFilter(filter, filterConditions.newInstance(XmpFilterMode.FILTER));
            }
            return new XMLEventReaderInputStream(filter, getOutputFactory());
        } catch (XMLStreamException ex) {
            throw new IOException(ex);
        }
    }

    @Override
    public InputStream sieve(InputStream xmpIS) throws IOException {
        log.debug("sieve");
        /* Sieve: !in(whitelist) || in(blacklist) */
        XMLInputFactory xmlif = getInputFactory();
        try {
            XMLEventFilter filter = new XMLReaderEventFilter(xmlif.createXMLEventReader(xmpIS));
            if (sieveConditions == null) {
                sieveConditions = parseNameList("", false, XmpFilterMode.SIEVE); /* accept none */
            }
            filter = new XmpPropFilter(filter, sieveConditions.newInstance(XmpFilterMode.SIEVE));
            return new XMLEventReaderInputStream(filter, getOutputFactory());
        } catch (XMLStreamException ex) {
            throw new IOException(ex);
        }
    }

    private XMLInputFactory getInputFactory() {
        XMLInputFactory xmlif = XMLInputFactory.newFactory();
        xmlif.setProperty(
                XMLInputFactory.IS_SUPPORTING_EXTERNAL_ENTITIES,
                Boolean.FALSE);
        xmlif.setProperty(XMLInputFactory.IS_NAMESPACE_AWARE,
                Boolean.TRUE);
        xmlif.setProperty(XMLInputFactory.IS_COALESCING,
                Boolean.TRUE);
        xmlif.setXMLResolver(DEFAULT_XML_RESOLVER);
        return xmlif;
    }

    private XMLOutputFactory getOutputFactory() {
        XMLOutputFactory xmlof = XMLOutputFactory.newFactory();
        xmlof.setProperty(XMLOutputFactory.IS_REPAIRING_NAMESPACES, Boolean.TRUE);
        return xmlof;
    }

    private enum XmpFilterState {
        LOOKING_FOR_RDF,
        INSIDE_RDF
    }

    interface XMLEventFilter {
        XMLEvent nextEvent() throws XMLStreamException;

        boolean hasNext() throws XMLStreamException;

        void close() throws XMLStreamException;
    }

    private static class XMLReaderEventFilter implements XMLEventFilter {
        private final XMLEventReader reader;
        private XMLEvent nextEvent;

        XMLReaderEventFilter(XMLEventReader reader) {
            this.reader = reader;
        }

        public XMLEvent nextEvent() throws XMLStreamException {
            if (hasNext()) {
                XMLEvent ev = nextEvent;
                nextEvent = null;
                return ev;
            }
            throw new XMLStreamException("no more events available");
        }

        public boolean hasNext() throws XMLStreamException {
            while (nextEvent == null) {
                if (!this.reader.hasNext()) {
                    break;
                }
                nextEvent = reader.nextEvent();
            }
            return nextEvent != null;
        }

        @Override
        public void close() throws XMLStreamException {
            reader.close();
        }
    }

    private class XmpPropFilter implements XMLEventFilter {

        private final EventConditionProvider conditions;

        private final XMLEventFilter filter;
        private final List<XMLEvent> nextEvents;
        private XmpFilterState state;
        private EventCondition eventCondition;
        private Stack<EventCondition> conditionStack;
        private Set<QName> scratchNames;
        private boolean accepted;

        private int level;
        private int ignoreLevel;

        XmpPropFilter(XMLEventFilter filter, EventConditionProvider conditions) {
            this.filter = filter;
            this.conditions = conditions;
            this.nextEvents = new ArrayList<XMLEvent>();
            this.eventCondition = EventCondition.ACCEPT;
            this.scratchNames = new HashSet<QName>();
            this.conditionStack = new Stack<EventCondition>();
            // protect against empty stack for ill-formed XML.
            conditionStack.push(EventCondition.DENY);
            this.accepted = true;
            this.ignoreLevel = Integer.MAX_VALUE;
            this.state = XmpFilterState.LOOKING_FOR_RDF;

            this.level = 0;
        }

        public XMLEvent nextEvent() throws XMLStreamException {
            if (hasNext()) {
                return nextEvents.remove(0);
            }
            throw new XMLStreamException("no more events available");
        }

        public boolean hasNext() throws XMLStreamException {
            while (nextEvents.isEmpty()) {
                if (!filter.hasNext()) {
                    break;
                }
                process(filter.nextEvent());
            }
            return !nextEvents.isEmpty();
        }

        private void process(XMLEvent event) {
            /**
             * All the tags outside of rdf:RDF are accepted.
             * Start filtering on configured conditions from rdf:RDF tag.
             *
             * On START_ELEMENT event:
             * Get configured condition for tags other then array containers[rdf:Alt, rdf:Bag, rdf:Seq] and rdf:li .
             * (For array containers[rdf:Alt, rdf:Bag, rdf:Seq] and rdf:li keep their parent's condition)
             * Evaluate the condition and push it to a stack.
             * if condition was accepted then apply attribute conditions.
             * Set an ignoreLevel once any condition is rejected, Don't evaluate conditions for further levels.
             *
             * On END_ELEMENT event: Pop the condition from stack and evaluate.
             * Reset ignoreLevel once the level ends (level == ignoreLevel).
             */

            switch (event.getEventType()) {
                case XMLEvent.START_ELEMENT:
                    QName name = event.asStartElement().getName();

                    switch (state) {
                        case LOOKING_FOR_RDF:
                            if (N_RDF.equals(name)) {
                                state = XmpFilterState.INSIDE_RDF;
                                eventCondition = EventCondition.ACCEPT;
                                level++;
                            }
                            break;
                        case INSIDE_RDF:
                            if (level <= ignoreLevel) {
                                if (RdfContainerNames.contains(name) || N_LI.equals(name)) {
                                    eventCondition = conditionStack.peek();
                                } else {
                                    eventCondition = conditions.getCondition(name);
                                }
                            }
                           // no state change
                            level++;
                            break;
                    }

                    int n = nextEvents.size();

                    // push and evaluate event condition but not for level higher then ignore level
                    if (level <= ignoreLevel) {
                        conditionStack.push(eventCondition);
                        accepted = eventCondition.accepts(event, nextEvents);

                        // set ignore level on rejected condition
                        if (!accepted) {
                            ignoreLevel = level;
                        }
                    } else {
                        // ignore event
                    }

                    if (!accepted) {
                        if (log.isDebugEnabled()) {
                            log.debug("excluded: {}", name);
                        }
                    } else {
                        // Apply attribute filtering conditions
                        if (state == XmpFilterState.INSIDE_RDF) {
                            filterPropertyAttributesAt(n);
                        }
                    }
                    break;

                case XMLEvent.END_ELEMENT:
                    name = event.asEndElement().getName();

                    if (XmpFilterState.INSIDE_RDF == state && N_RDF.equals(name)) {
                        state = XmpFilterState.LOOKING_FOR_RDF;
                        eventCondition = EventCondition.ACCEPT;
                    }

                    if (level <= ignoreLevel) {
                        eventCondition = conditionStack.pop();
                        accepted = eventCondition.accepts(event, nextEvents);
                    } else {
                        // ignore event
                    }

                    // reset ignoreLevel, once the tag (where ignore level started) is ended.
                    if (level == ignoreLevel) {
                        ignoreLevel = Integer.MAX_VALUE;
                    }
                    level--;

                    break;

                default:
                    if (level < ignoreLevel) {
                        EventCondition.ACCEPT.accepts(event, nextEvents);
                    }
                    break;
            }
        }

        private void filterPropertyAttributesAt(int index) {
            XMLEvent event = nextEvents.get(index);
            if (event.getEventType() == XMLEvent.START_ELEMENT) {
                StartElement se = event.asStartElement();
                scratchNames.clear();
                @SuppressWarnings("rawtypes") // due to rawtypes contract of javax.xml.stream.events
                Iterator attrIter = se.getAttributes();
                while (attrIter.hasNext()) {
                    Attribute attr = (Attribute) attrIter.next();
                    QName aname = attr.getName();
                    if (!IgnoredPropertyAttributeNamespaces.contains(aname.getNamespaceURI())
                            && !IgnoredPropertyAttributeNames.contains(aname)) {
                        EventCondition cond = conditions.getCondition(aname);
                        if (!cond.acceptsValue(attr.getValue())) {
                            scratchNames.add(aname);
                            if (log.isDebugEnabled()) {
                                log.debug("excluded: {}", aname);
                            }
                        }
                    }
                }
                if (!scratchNames.isEmpty()) {
                    event = new StartElementWrapper(se, scratchNames);
                    scratchNames.clear();
                    nextEvents.remove(index);
                    nextEvents.add(index, event);
                }
            }
        }

        public void close() throws XMLStreamException {
            filter.close();
        }
    }

    private final class XMLEventReaderInputStream extends InputStream {

        final XMLEventWriter writer;
        final XMLEventFilter filter;
        final ReadableByteArrayOutputStream output;
        boolean closed;

        XMLEventReaderInputStream(XMLEventFilter filter, XMLOutputFactory xmlof)
                throws XMLStreamException {
            this.filter = filter;
            this.output = new ReadableByteArrayOutputStream(4 * 1024);
            this.writer = xmlof.createXMLEventWriter(this.output, "utf-8");
            this.closed = false;
        }

        @Override
        public int available() throws IOException {
            return output.size();
        }

        @Override
        public void close() throws IOException {
            super.close();
            if (!closed) {
                try {
                    filter.close();
                } catch (XMLStreamException ex2) {
                    log.debug("closing filter", ex2);
                }
                try {
                    writer.close();
                } catch (XMLStreamException ex2) {
                    log.debug("closing writer", ex2);
                }
                closed = true;
            }
        }

        @Override
        public int read() throws IOException {
            prefill(1);
            byte[] x = new byte[1];
            int len = output.read(x, 0, 1);
            return (len > 0) ? x[0] : -1;
        }

        @Override
        public int read(byte[] b, int off, int len) throws IOException {
            prefill(len);
            len = output.read(b, off, len);
            return (len > 0 || !closed) ? len : -1;
        }

        private void prefill(int length) throws IOException {
            try {
                while (!closed && output.size() < length) {
                    if (!filter.hasNext()) {
                        close();
                        break;
                    }
                    writer.add(filter.nextEvent());
                }
            } catch (XMLStreamException ex) {
                throw new IOException(ex);
            }
        }
    }

    private final class ReadableByteArrayOutputStream extends ByteArrayOutputStream {

        ReadableByteArrayOutputStream(int size) {
            super(size);
        }

        int read(byte[] b, int off, int len) throws IOException {
            if (len > super.count) {
                len = super.count;
            }
            if (len > 0) {
                int remain = count - len;
                System.arraycopy(super.buf, 0, b, off, len);
                if (remain > 0) {
                    System.arraycopy(super.buf, len, super.buf, 0, remain);
                }
                super.count -= len;
            }
            return len;
        }
    }

    private final static class StartElementWrapper implements StartElement {

        private final StartElement se;
        private final Map<QName, Attribute> attributes;

        StartElementWrapper(StartElement se, Set<QName> removeAttributes) {
            this.se = se;
            this.attributes = new HashMap<QName, Attribute>();
            @SuppressWarnings("rawtypes") // due to rawtypes contract of javax.xml.stream.events
            Iterator attrIter = se.getAttributes();
            while (attrIter.hasNext()) {
                Attribute attr = (Attribute) attrIter.next();
                if (!removeAttributes.contains(attr.getName())) {
                    attributes.put(attr.getName(), attr);
                }
            }
        }

        @Override
        public QName getName() {
            return se.getName();
        }

        @SuppressWarnings("rawtypes") // due to rawtypes contract of javax.xml.stream.events
        @Override
        public Iterator getAttributes() {
            return attributes.values().iterator();
        }

        @SuppressWarnings("rawtypes") // due to rawtypes contract of javax.xml.stream.events
        @Override
        public Iterator getNamespaces() {
            return se.getNamespaces();
        }

        @Override
        public Attribute getAttributeByName(QName name) {
            return attributes.get(name);
        }

        @Override
        public NamespaceContext getNamespaceContext() {
            return se.getNamespaceContext();
        }

        @Override
        public String getNamespaceURI(String prefix) {
            return se.getNamespaceURI(prefix);
        }

        @Override
        public int getEventType() {
            return se.getEventType();
        }

        @Override
        public Location getLocation() {
            return se.getLocation();
        }

        @Override
        public boolean isStartElement() {
            return se.isStartElement();
        }

        @Override
        public boolean isAttribute() {
            return se.isAttribute();
        }

        @Override
        public boolean isNamespace() {
            return se.isNamespace();
        }

        @Override
        public boolean isEndElement() {
            return se.isEndElement();
        }

        @Override
        public boolean isEntityReference() {
            return se.isEntityReference();
        }

        @Override
        public boolean isProcessingInstruction() {
            return se.isProcessingInstruction();
        }

        @Override
        public boolean isCharacters() {
            return se.isCharacters();
        }

        @Override
        public boolean isStartDocument() {
            return se.isStartDocument();
        }

        @Override
        public boolean isEndDocument() {
            return se.isEndDocument();
        }

        @Override
        public StartElement asStartElement() {
            return this;
        }

        @Override
        public EndElement asEndElement() {
            return se.asEndElement();
        }

        @Override
        public Characters asCharacters() {
            return se.asCharacters();
        }

        @Override
        public QName getSchemaType() {
            return se.getSchemaType();
        }

        @Override
        public void writeAsEncodedUnicode(Writer writer) throws XMLStreamException {
            se.writeAsEncodedUnicode(writer);
        }
    }
}
