package com.day.cq.dam.commons.watermark;

import com.day.image.Layer;

import java.awt.*;
import java.awt.font.FontRenderContext;
import java.awt.font.TextAttribute;
import java.awt.font.TextLayout;
import java.awt.geom.Rectangle2D;
import java.awt.image.BufferedImage;
import java.text.AttributedCharacterIterator;
import java.text.AttributedString;

/**
 *  A {@link TextWatermark} represents a string of text to be used to watermark an underlying image.<br>
 *  While actually applying the watermark effect,
 *  the styled text is converted to an image so that the effect is uniform.
 */
public class TextWatermark extends ImageWatermark {

    public static final String COPYRIGHT = "\u00a9";

    private String text;

    private Font font;

    private boolean vertical;

    public TextWatermark(String text) {
        this.text = text;
        this.font = new Font();
    }

    public TextWatermark(String text, Font font) {
        this.text = text;
        this.font = font;
    }

    public TextWatermark(Location position, double orientation, float opacity, String text, Font font) {
        super(position, orientation, opacity);
        this.text = text;
        this.font = font;
    }

    public TextWatermark(int top, int left, double orientation, float opacity, String text, Font font) {
        super(top, left, orientation, opacity);
        this.text = text;
        this.font = font;
    }

    public TextWatermark(Location position, String text, Font font) {
        super(position);
        this.text = text;
        this.font = font;
    }

    public TextWatermark(int top, int left, String text, Font font) {
        super(top, left);
        this.text = text;
        this.font = font;
    }

    public String getText() {
        return text;
    }

    public void setText(String text) {
        this.text = text;
    }

    public Font getFont() {
        return font;
    }

    public void setFont(Font font) {
        this.font = font;
    }

    /**
     * Generates a transparent image representing the text of this watermark.
     * Text in the image will abide by the font specifications.
     *
     * @return  the {@link BufferedImage} object thus produced
     */
    @Override
    public BufferedImage getImage() {
        //create the font you wish to use
        java.awt.Font textFont = new java.awt.Font(this.getFont().getFamily(), java.awt.Font.PLAIN, this.font.getSize());

        //create the FontRenderContext object which helps us to measure the text
        FontRenderContext frc = new FontRenderContext(null, true, true);

        //get the height and width of the text
        Rectangle2D bounds = textFont.getStringBounds(this.text, frc);

        int w = 0;
        int h = 0;
        if(this.vertical) {
            for(int i=0; i<this.text.length(); i++) {
                String character = this.text.substring(i, i+1);
                Rectangle2D charBounds = textFont.getStringBounds(character, frc);
                h += (int) charBounds.getHeight();
                w = (int) charBounds.getWidth() > w ? (int) charBounds.getWidth() : w;
            }
        }
        else {
            w = (int) bounds.getWidth();
            h = (int) bounds.getHeight();
        }

        //create a BufferedImage object
        BufferedImage image = new BufferedImage(w, h, BufferedImage.TYPE_INT_ARGB);
        //calling createGraphics() to get the Graphics2D
        Graphics2D g = image.createGraphics();

        g.setColor(new Color(255,255,255,0));
        g.fillRect(0, 0, w, h);

        //make background transparent using alpha=0
        g.setColor(new Color(255,255,255,0));
        g.fillRect(0, 0, w, h);

        //prepare to write using font settings
        g.setColor(this.font.getColor());
        g.setFont(textFont);

        //write the actual string onto the background
        if(this.vertical) {
            float x = (float) bounds.getX();
            float y = (float) -bounds.getY();
            int charHeight = 0;
            for(int i=0; i<this.text.length(); i++) {
                String character = this.text.substring(i, i + 1);
                Rectangle2D charBounds = textFont.getStringBounds(character, frc);
                y += charHeight;
                g.drawString(character, x, y);
                charHeight = (int) charBounds.getHeight();
            }
        }
        else {
            g.drawString(this.text, (float) bounds.getX(), (float) -bounds.getY());
        }

        //releasing resources
        g.dispose();

        return image;
    }

    @Override
    public String toString() {
        return  super.toString() +
                "\ntext = " + text +
                "\nfont = " + font.toString();
    }

    public boolean isVertical() {
        return vertical;
    }

    public void setVertical(boolean vertical) {
        this.vertical = vertical;
    }
}
