/*
 * Copyright 1997-2004 Day Management AG
 * Barfuesserplatz 6, 4001 Basel, Switzerland
 * All Rights Reserved.
 *
 * This software is the confidential and proprietary information of
 * Day Management AG, ("Confidential Information"). You shall not
 * disclose such Confidential Information and shall use it only in
 * accordance with the terms of the license agreement you entered into
 * with Day.
 */
package com.day.cq.graphics.chart;

import com.day.cq.graphics.Graph;
import com.day.image.Layer;

/**
 * The <code>PieChart</code> class implements drawing a pie chart from data
 * samples. From each data row only the first sample is taken to draw the pie.
 * The slices are drawn counter clock wise, last row to first starting at the
 * <code>start</code> angle as defined with {@link #setStart(double)}.
 * <p/>
 * As a side effect the the <code>ticksample</code> data in the Y axis of the
 * graph is set to the percentage of the sum of sample values. See
 * {@link #draw(Graph, boolean)} for more information.
 *
 * @author fmeschbe
 */
public class PieChart extends Chart {

    /**
     * The width of the pie chart
     */
    int width;

    /**
     * The height of the pie chart
     */
    int height;

    /**
     * The starting angle of the pie chart in degrees, 0 being 3 o'clock
     */
    double start;

    public static String getName() {
        return "pie";
    }

    /**
     * Draws the pie chart into a new layer of the defined height and width. As
     * noted in the above class comment, this method sets the
     * <code>ticksample</code> value of the graph's Y axis as the chart is
     * drawn. This method can thus be used to simply calculate this data.
     *
     * @param graph  The graph the chart belongs to
     * @param doDraw Indicates whether to draw or only calculate
     * @return The layer into which the pie chart has been drawn or null if no
     *         drawing has been requested.
     */
    public Layer draw(Graph graph, boolean doDraw) {

        // get geometrics and create layer
        Layer l = graph.createLayer(doDraw, width, height);

        int r = ((width < height) ? width : height) / 2;
        int mx = r;
        int my = r;

        // init values
        double from = start;
        Data data = graph.getData();
        Axis yAxis = graph.getYAxis();
        int num = data.getNumrows();

        // get sum of all sclices
        double sum = 0;
        for (int i = 0; i < num; i++) sum += data.getDataRow(i).samples[0];

        for (int k = 0, i = num - 1; i >= 0; i--, k++) {

            DataRow row = data.getDataRow(k);
            double percent = row.samples[0] / sum;

            yAxis.ticksample[k] = 100 * percent;

            if (doDraw) {
                l.setPaint(row.color);
                if (percent == 1) {
                    l.fillEllipse(mx, my, r, r);
                } else if (percent > 0) {
                    double extent = 360 * percent;
                    l.fillSector(mx, my, r, r, from, extent);
                    from += extent;
                }
            }
        }

        yAxis.ticksample[num] = 0;
        yAxis.numticks = num + 1;

        return l;
    }

    /**
     * Sets the width of the pie chart, if it has not been set yet.
     *
     * @param width The width of the pie chart.
     */
    public void setWidth(int width) {
        if (this.width == 0) this.width = width;
    }

    /**
     * Sets the height of the pie chart, if it has not been set yet.
     *
     * @param height The height of the pie chart.
     */
    public void setHeight(int height) {
        if (this.height == 0) this.height = height;
    }

    /**
     * Sets the starting angle of the first pie piece, if it has not been set
     * yet. This angle is considered to be in degrees, to rotate counter
     * clockwise for positive values. Zero degrees is at 3 o'clock.
     *
     * @param start
     */
    public void setStart(double start) {
        if (this.start == 0) this.start = start;
    }

}
