/*
 * Copyright 1997-2010 Day Management AG
 * Barfuesserplatz 6, 4001 Basel, Switzerland
 * All Rights Reserved.
 *
 * This software is the confidential and proprietary information of
 * Day Management AG, ("Confidential Information"). You shall not
 * disclose such Confidential Information and shall use it only in
 * accordance with the terms of the license agreement you entered into
 * with Day.
 */
package com.day.cq.mailer.commons;

import com.adobe.granite.security.user.UserProperties;
import org.apache.commons.lang.text.StrLookup;
import org.apache.jackrabbit.api.security.user.Authorizable;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.jcr.RepositoryException;
import java.beans.IntrospectionException;
import java.beans.PropertyDescriptor;
import java.lang.reflect.Method;
import java.util.HashMap;
import java.util.Map;

import static java.beans.Introspector.IGNORE_ALL_BEANINFO;
import static java.beans.Introspector.getBeanInfo;

/**
 * Implements a {@link org.apache.commons.lang.text.StrLookup StrLookup} that is
 * looking up variables from a
 * {@link org.apache.jackrabbit.api.security.user.Authorizable Authorizable}'s UserProperties.
 * 
 * Additional variables can bee added via a {@link #put(String, String)}.
 * If the key exists in the UserProperties, the value is overwritten with the newly set
 * @see StrLookup
 * @since 5.4
 */
public class ProfileVariableLookup extends StrLookup {

    public final static String AUTHORIZABLE_ID = "authorizableID";

    private final static Logger log = LoggerFactory.getLogger(ProfileVariableLookup.class);

    private final Map<String, String> overlay;

    private final Map<String, Method> map;

    private final Authorizable authorizable;

    private final UserProperties userProperties;

    /**
     * 
     * @param authorizable The authorizable for which variables need to be looked up.
     * @param userProperties The properties of the authorizable.
     * @param overlays The map of overlays
     * @throws IntrospectionException when not being able to map a string class name to a Class object
     */
    public ProfileVariableLookup(Authorizable authorizable, UserProperties userProperties, Map<String, String> overlays) throws IntrospectionException {
        this.authorizable = authorizable;
        try {
            if (authorizable != null && authorizable.getID() != null ) {
                overlays.put("userUUID", authorizable.getID());
            }
        } catch (RepositoryException e) {
            log.warn("Could not read authorizable ID to replace 'userUUID' in email", e);
        }

        this.map = new HashMap<String, Method>();
        this.userProperties = userProperties;
        //note this lines release the dependency to a commons.bean
        for (PropertyDescriptor propDesc : getBeanInfo(authorizable.getClass(), IGNORE_ALL_BEANINFO).getPropertyDescriptors()) {
            if (propDesc.getPropertyType().isAssignableFrom(String.class)) {
                map.put(propDesc.getName(), propDesc.getReadMethod());
            }
        }
        this.overlay = overlays;

    }

    /**
     * {@inheritDoc}
     */
    @Override
    public String lookup(String variable) {
        if (overlay.containsKey(variable)) {
            return overlay.get(variable);
        }
        //add the id to the variable as there is no getter.
        if (AUTHORIZABLE_ID.equals(variable)) {
            if (userProperties != null) {
                return userProperties.getAuthorizableID();
            }
        }
        if (authorizable != null ) {
            try {
                if(authorizable.hasProperty(variable) )
                    return authorizable.getProperty(variable)[0].getString();
            } catch (Exception e) {
                log.error(e.getMessage());

            }
        }

        if(userProperties != null){
            try {
                if(userProperties.getProperty(variable) != null){
                    return userProperties.getProperty(variable);
                }
            } catch (RepositoryException e) {
                log.error(e.getMessage());
            }
        }
        return null;
    }

    /**
     * Adds or overwrites a mapping of this replacer.
     * @param var name of the variable to map
     * @param replace the value of the variable
     */
    public void put(String var, String replace) {
        overlay.put(var, replace);
    }
}