/*
 * Copyright 1997-2008 Day Management AG
 * Barfuesserplatz 6, 4001 Basel, Switzerland
 * All Rights Reserved.
 *
 * This software is the confidential and proprietary information of
 * Day Management AG, ("Confidential Information"). You shall not
 * disclose such Confidential Information and shall use it only in
 * accordance with the terms of the license agreement you entered into
 * with Day.
 */
package com.day.cq.preferences;

import java.util.Collection;

import javax.jcr.AccessDeniedException;
import javax.jcr.PathNotFoundException;

/**
 * Preferences are a collection of properties similar to a {@link java.util.Dictionary Dictionary}.
 * Preferences are configuration properties for a particular user of the application.
 * Like Properties Preferences have a name and a value, which can be any
 * {@link Object Object}.<br>
 * In addtion to Properties, Preferences may contain further Preferences,
 * which are maintained in hierarchical structure.
 * <b>But</b> Properties do <b>NOT</b> add any further semantic to the values
 * nor to the hierarchy.<p>
 * The current specification differs from other Preferences like the
 * {@link java.util.prefs.Preferences Java Preferences} or the OSGI's
 * by the following:<br>
 * This Preferences do not impose implicit categorization like the
 * Java package in case of Java Preferences resp. the Bundle's name for the OSGI
 * Preferences.
 * As benefit of this behavior, Preferences can be organized according
 * their inherent structure rather than than the structure of source code.
 *
 * @see PreferencesService
 * @deprecated cq 5.5
 */
public interface Preferences {

    /**
     * @return Path of the resource this preference is persisted at
     */
    String getPath();

    /**
     * @return the AuthorizableId this Preferences are created for
     * @see PreferencesService#get(String)
     */
    String getAuthorizableId();

    /**
     * Access the relative Path of the current Preferences within the hierarchy.
     * The value has to be in format, that
     * <pre>
     *  String path = "a/b";
     *  Preferences rootPref = preferencesService.get(session)
     *  rootPref.getPreferences(path).getPath().equals(path)
     * </pre>
     *
     * @return path to the current preferences within the Preferences Hierarchy
     */
    String getPreferenceName();

    /**
     * The names of all Preferences which contain a value.
     * The value may be set directly on this Authorizable or inherited.
     * NOTE: this does not mean, that this is the possible exhaustive set of
     * Preferences names
     *
     * @return all Preference names that have a value
     */
    Collection<String> getPreferencesNames();

    /**
     * All direct children Preferences. The may be set directly on this instance
     * or on an inherited one.
     * NOTE: this does not mean, that this is the possible exhaustive set of
     * children
     *
     * @return all children Preferences that are set
     */
    Collection<Preferences> getChildren();

    /**
     * Return Preferences contained with in this Preferences at the given relative
     * Path.<br>
     * A <code>null</code> Preference is not allowed as return value.
     * If there are no Preferences at the given Path, implementations
     * have to create new empty Preferences or throw a
     * {@link javax.jcr.PathNotFoundException PathNotFoundException}, if this
     * is not possible.<br>
     * Thus this method servers as a Factory for new Preferences to be contained
     * in this ones
     *
     * @param relativePath relative relativePath within the current Preferences.
     * @return Preferences
     * @throws PathNotFoundException if the Path is absolute or not contained
     *                               within this Preferences. If path points to a Preferences value
     *                               other than of type Preferences.
     *                               A denied access to Preferences has to result in a PathNotFoundException
     */
    Preferences getPreferences(String relativePath) throws PathNotFoundException;

    /**
     * Shortcut access to a String Preference.
     *
     * @param relativePath of the Preference
     * @return the String representation of the value
     * @throws PathNotFoundException if the path does not exist, does not point to
     *                               Preferences-value that may be converted to a String, or
     *                               access restrictions prevent from accessing the value.
     * @see #get(String, Object)
     */
    String get(String relativePath) throws PathNotFoundException;

    /**
     * Access a Preference value at a given Path and type given by means of the
     * default value.
     * If the current Preferences don't have a value for the given path, the default
     * value is returned.
     *
     * @param relativePath path to Preference
     * @param defaultValue value to take return type or
     *                     value if none exists at the given path
     * @return <code>none-null</code> value of given Type
     * @throws PathNotFoundException if path is not accessible for any reason
     */
    <Type> Type get(String relativePath, Type defaultValue) throws PathNotFoundException;

    /**
     * @param relativePath path to preference
     * @param defaultValue to take if none is contained in the current preferences
     * @param type         the value has to be returned
     * @return the preference if one set or the default value
     * @throws PathNotFoundException in any case the path can not be accessed
     */
    <Type> Type get(String relativePath, Type defaultValue, Class<Type> type)
            throws PathNotFoundException;

    /**
     * Store the given value at the given path.
     * Implementations are not required to persist changes immediately.
     *
     * @param relativePath path to store the given value at
     * @param value        to store
     * @throws AccessDeniedException if access restrictions prevent from saving the value
     * @see #flush()
     */
    void put(String relativePath, Object value) throws AccessDeniedException;

    /**
     * Remove the value at the given path.
     * Method need not signal an exceptional state if the given path does not exist
     * or is invalid.<br>
     * Implementations are not required to persist changes immediately.
     *
     * @param relativePath to the preference to remove
     * @throws AccessDeniedException if access restrictions prevent from removing a Preference
     * @see #flush()
     */
    void remove(String relativePath) throws AccessDeniedException;

    /**
     * A call to this method synchronizes the values of this Preferences with its
     * persistent state.
     *
     * @see #flush()
     */
    void refresh();

    /**
     * This method persists all changes made to this instance.
     *
     * @throws AccessDeniedException in case access restrictions prevent from
     *                               writing to persistence layer.
     */
    void flush() throws AccessDeniedException;

}
