/*
 * Copyright 1997-2010 Day Management AG
 * Barfuesserplatz 6, 4001 Basel, Switzerland
 * All Rights Reserved.
 *
 * This software is the confidential and proprietary information of
 * Day Management AG, ("Confidential Information"). You shall not
 * disclose such Confidential Information and shall use it only in
 * accordance with the terms of the license agreement you entered into
 * with Day.
 */

package com.day.cq.reporting;

import java.util.HashMap;
import java.util.Map;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * This class implements a row of {@link Data}.
 */
public class DataRow {

    /**
     * Logger
     */
    private final Logger log = LoggerFactory.getLogger(DataRow.class);

    /**
     * Map that contains the cell data; key: property; value: the value
     */
    private final Map<String, CellValue> cellData;

    /**
     * Flag that determines if the row has already been compacted
     */
    private boolean isCompacted;


    /**
     * Creates a new row of report data.
     *
     * @param colCnt Number of columns (should be set appropriately to save memory)
     */
    public DataRow(int colCnt) {
        this.cellData = new HashMap<String, CellValue>(colCnt);
        this.isCompacted = false;
    }

    /**
     * Ensures that the data is still mutable. Throws an {@link IllegalStateException}
     * otherwise.
     */
    protected void ensureMutable() {
        if (this.isCompacted) {
            throw new IllegalStateException(
                    "Report data is already compacted and therefore immutable.");
        }
    }

    /**
     * Compacts row data
     */
    protected void compact() {
        this.ensureMutable();
        this.isCompacted = true;
    }

    /**
     * <p>Adds cell data.</p>
     *
     * <p>If there's already cell data available for the specified property, the existing
     * cell data gets overwritten by the new value.</p>
     *
     * @param property Name of the property
     * @param value The property's value for the eow
     */
    public void add(String property, CellValue value) {
        this.ensureMutable();
        log.debug("Setting property '{}' to value '{}'", property, value);
        this.cellData.put(property, value);
    }

    /**
     * Gets the data of the specified cell.
     *
     * @param property Name of property for which the data has to be retrieved
     * @return The property's value for the row; <code>null</code> if no value is available
     */
    public CellValue get(String property) {
        return this.cellData.get(property);
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public String toString() {
        StringBuilder dump = new StringBuilder(128);
        boolean isFirst = true;
        for (String key : this.cellData.keySet()) {
            if (isFirst) {
                isFirst = false;
            } else {
                dump.append("; ");
            }
            dump.append("[ ").append(key).append(" = (");
            dump.append(this.cellData.get(key));
            dump.append(") ]");
        }
        return dump.toString();
    }

}
