/*
 * Copyright 1997-2010 Day Management AG
 * Barfuesserplatz 6, 4001 Basel, Switzerland
 * All Rights Reserved.
 *
 * This software is the confidential and proprietary information of
 * Day Management AG, ("Confidential Information"). You shall not
 * disclose such Confidential Information and shall use it only in
 * accordance with the terms of the license agreement you entered into
 * with Day.
 */

package com.day.cq.reporting;

/**
 * This enumeration defines the types of data processing phases available.
 */
public enum ProcessingPhase {

    /**
     * Apply the preprocessing to the raw data (keeping it all the way through processing)
     */
    APPLY("apply", true),

    /**
     * Apply the preprocessing to the completely processed data (grouped, aggregated, etc.;
     * the processed data will be replaced accordingly)
     */
    APPLY_AFTER("applyAfter", true),

    /**
     * Apply the preprocessing for grouping. The raw data will remain unchanged.
     */
    GROUP("group", false),

    /**
     * Apply the preprocessing for filtering. The raw data will remain unchanged.
     */
    FILTER("filter", false);

    /**
     * The string representation
     */
    private String stringRep;

    /**
     * Flag that determines if the value should be applied to the data
     */
    private boolean applyValue;


    /**
     * Creates a new enumeration value.
     *
     * @param stringRep The string representation
     * @param applyValue <code>true</code> if the preprocessed value should be applied
     *                   to the data, replacing the raw data variant
     */
    private ProcessingPhase(String stringRep, boolean applyValue) {
        this.stringRep = stringRep;
        this.applyValue = applyValue;
    }

    /**
     * Gets the string representation of the preprocessing module.
     *
     * @return The string representation of the preprocessing module
     */
    public String getStringRep() {
        return this.stringRep;
    }

    /**
     * Checks if the formatted value should be applied to the data (versus being used for
     * calculation only).
     *
     * @return <code>true</code> if the formatted value should be applied to the data
     */
    public boolean applyValue() {
        return this.applyValue;
    }

    /**
     * Creates a suitable <code>ProcessingPhase</code> from the specified string
     * representation.
     *
     * @param stringRep The string representation
     * @return The preprocessing type; <code>null</code> if an invalid string representation
     *         has been specified
     */
    public static ProcessingPhase fromStringRep(String stringRep) {
        ProcessingPhase[] types = values();
        for (ProcessingPhase type : types) {
            if (type.getStringRep().equals(stringRep)) {
                return type;
            }
        }
        return null;
    }

}
