/*
 * Copyright 1997-2008 Day Management AG
 * Barfuesserplatz 6, 4001 Basel, Switzerland
 * All Rights Reserved.
 *
 * This software is the confidential and proprietary information of
 * Day Management AG, ("Confidential Information"). You shall not
 * disclose such Confidential Information and shall use it only in
 * accordance with the terms of the license agreement you entered into
 * with Day.
 */
package com.day.cq.rewriter.linkchecker;

import java.net.URI;
import java.net.URISyntaxException;

import javax.jcr.Item;
import javax.jcr.Node;
import javax.jcr.RepositoryException;

import org.apache.sling.api.SlingHttpServletRequest;
import org.apache.sling.api.resource.Resource;
import org.apache.sling.api.resource.ResourceResolver;
import org.apache.sling.api.resource.ResourceUtil;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.day.cq.commons.jcr.JcrConstants;

/**
 * Settings that control the behaviour of the {@link LinkChecker}. The settings
 * are used as a "linkchecker context" that holds the base-ref and
 * context uri, and to control the actual rewrite behavior.
 *
 * If "request" is not attached, they are used merely temporary before the
 * actual linkchecker is initialized.
 */
public class LinkCheckerSettings {

    /**
     * default logger
     */
    private static final Logger log = LoggerFactory.getLogger(LinkCheckerSettings.class);

    /**
     * the name of the request attribute
     */
    public static final String REQUEST_ATTRIBUTE_NAME = LinkCheckerSettings.class.getName();

    /**
     * The HTTP request whose response we are checking.
     */
    private SlingHttpServletRequest request;

    /**
     * rewrite config for invalid links
     */
    private LinkRewriteConfig invalidConfig;

    /**
     * rewrite config for expired links
     */
    private LinkRewriteConfig expiredConfig;

    /**
     * rewrite config for predated links
     */
    private LinkRewriteConfig predatedConfig;

    /**
     * The base url to resolve relative to.
     */
    private String baseRef;

    /**
     * if <code>true</code> internal links are not checked
     */
    private boolean ignoreInternals;

    /**
     * if <code>true</code> internal links are not checked
     */
    private boolean ignoreExternals;

    /**
     * the cached context uri
     */
    private URI contextUri;

    /**
     * the cached base uri
     */
    private URI baseUri;

    /**
     * Returns the current link checker settings for the current request.
     *
     * @param request the request
     * @return the settings or <code>null</code>
     */
    public static LinkCheckerSettings fromRequest(SlingHttpServletRequest request) {
        LinkCheckerSettings settings = (LinkCheckerSettings) request.getAttribute(REQUEST_ATTRIBUTE_NAME);
        if (settings == null) {
            // create temporary settings
            settings = new LinkCheckerSettings();
            request.setAttribute(REQUEST_ATTRIBUTE_NAME, settings);
        }
        return settings;
    }

    /**
     * Attach the settings to the request.
     *
     * @param request The current request
     */
    public void init(SlingHttpServletRequest request) {
        this.request = request;
    }

    public SlingHttpServletRequest getRequest() {
        return request;
    }

    /**
     * Returns the resource resolver
     *
     * @return the resource resolver
     */
    public ResourceResolver getResourceResolver() {
        return request.getResourceResolver();
    }

    /**
     * Returns the context path
     *
     * @return the context path
     */
    public String getContextPath() {
        return request.getContextPath();
    }

    /**
     * Returns the context URI of the web application associated with the
     * given settings.
     *
     * @return context URI, or <code>null</code> if the URI malformed
     * 
     * @since 5.3
     */
    public URI getContextURI() {
        if (contextUri == null) {
            try {
                String context = request.getContextPath() + "/";
                URI uri = new URI(request.getRequestURL().toString());
                contextUri = new URI(
                        uri.getScheme(), uri.getRawUserInfo(),
                        uri.getHost(), uri.getPort(), context, null, null);
            } catch (URISyntaxException e) {
                log.warn("Ignoring malformed context URI: {}", e.toString());
            }
        }
        return contextUri;
    }

    /**
     * Returns the config for invalid links
     *
     * @return the config for invalid links
     */
    public LinkRewriteConfig getInvalidConfig() {
        return invalidConfig;
    }

    /**
     * Sets the config for invalid links
     *
     * @param config the config
     */
    public void setInvalidConfig(LinkRewriteConfig config) {
        this.invalidConfig = config;
    }

    /**
     * Returns the config for expired links
     *
     * @return the config for expired links
     */
    public LinkRewriteConfig getExpiredConfig() {
        return expiredConfig;
    }

    /**
     * Sets the config for expired links
     *
     * @param config the config
     */
    public void setExpiredConfig(LinkRewriteConfig config) {
        this.expiredConfig = config;
    }

    /**
     * Returns the config for predated links
     *
     * @return the config
     */
    public LinkRewriteConfig getPredatedConfig() {
        return predatedConfig;
    }

    /**
     * Sets the config for predated links
     *
     * @param config the config
     */
    public void setPredatedConfig(LinkRewriteConfig config) {
        this.predatedConfig = config;
    }

    /**
     * Get the current request uri.
     *
     * @return the request uri
     */
    public String getRequestURI() {
        return request.getRequestURI();
    }

    /**
     * Return a uri to the parent of the current uri.
     *
     * @return the parent uri or an empty string
     */
    public String getParentRequestURI() {
        String requestURI = getRequestURI();
        if (requestURI == null) {
            return "";
        } else {
            String parentRequestURI = requestURI;

            // if the resource does not point to an item in the repository
            // we just take the immediate parent
            boolean doFallback = true;
            if (request.getResource().adaptTo(Item.class) != null) {
                final Resource parent = ResourceUtil.getParent(request.getResource());
                if (parent != null) {
                    Node n = parent.adaptTo(Node.class);
                    if (n != null) {
                        int index = 1;
                        try {
                            while (!n.isNodeType(JcrConstants.NT_HIERARCHYNODE)) {
                                n = n.getParent();
                                index++;
                            }
                            // we want the parent of the hierarchy node, so:
                            index++;
                            int pos = requestURI.length();
                            for (int i = 0; i < index; i++) {
                                pos = requestURI.lastIndexOf('/', pos - 1);
                            }
                            if (pos > 0) {
                                parentRequestURI = requestURI.substring(0, pos + 1);
                                doFallback = false;
                            }
                        } catch (RepositoryException e) {
                            // we ignore this!
                        }
                    }
                }
            }
            if (doFallback) {
                final int pos = requestURI.lastIndexOf('/');
                if (pos > 0) {
                    parentRequestURI = requestURI.substring(0, pos + 1);
                }
            }

            // now remove context prefix
            String prefix = getContextPath();
            if (prefix != null && prefix.length() > 0
                    && parentRequestURI.startsWith(prefix)) {
                if (parentRequestURI.length() == prefix.length()
                        || parentRequestURI.charAt(prefix.length()) == '/') {
                    return parentRequestURI.substring(prefix.length());
                }
            }

            return parentRequestURI;
        }
    }

    /**
     * Get the current base ref.
     *
     * @return the base href
     */
    public String getBaseRef() {
        return baseRef;
    }

    /**
     * Returns the base URI of the request associated with the given
     * settings.
     *
     * @return base URI, or <code>null</code> if the URI malformed
     *
     * @since 5.3
     */
    public URI getBaseURI() {
        if (baseUri == null) {
            String base = baseRef;
            if (base == null) {
                StringBuffer url = request.getRequestURL();
				if (url.indexOf("/jcr:content") > 0) {
					// links are relative to their page, not components on the page; see bug #36375
					base = url.substring(0, url.indexOf("/jcr:content"));
				} else {
                	base = url.toString();
				}
            }
            try {
                baseUri = new URI(base);
            } catch (URISyntaxException e) {
                log.warn("Ignoring malformed base URI: {}", e.toString());
                return null;
            }
        }
        return baseUri;
    }


    /**
     * Set the current base ref.
     *
     * @param baseRef the base href
     */
    public void setBaseRef(String baseRef) {
        this.baseRef = baseRef;
        this.baseUri = null;
    }

    /**
     * If <code>true</code> internal links are not checked
     * @return <code>true</code> if internal links are not checked
     */
    public boolean isIgnoreInternals() {
        return ignoreInternals;
    }

    /**
     * Controls if internal links should be checked by the linkchecker.
     * If set to <code>true</code> all internal links are treated as 'valid'.
     *
     * Please note that the settings are directly used by the linkchecker transformer
     * and are respected on a "startElement" event. When altered on-the-fly in a
     * JSP don't forget to flush the writer before changing the value.
     *
     * @param ignoreInternals if <code>true</code> internal links are not checked
     * @return previous value
     */
    public boolean setIgnoreInternals(boolean ignoreInternals) {
        boolean ret = this.ignoreInternals;
        this.ignoreInternals = ignoreInternals;
        return ret;
    }

    /**
     * If <code>true</code>, external links are not checked
     * @return <code>true</code> if external links are not checked
     */
    public boolean isIgnoreExternals() {
        return ignoreExternals;
    }

    /**
     * Controls if external links should be checked by the linkchecker.
     * If set to <code>true</code> all external links are treated as 'valid' and
     * they are also not recorded in the external-link storage.
     *
     * Please note that the settings are directly used by the linkchecker transformer
     * and are respected on a "startElement" event. When altered on-the-fly in a
     * JSP don't forget to flush the writer before changing the value.
     * 
     * @param ignoreExternals if <code>true</code> external links are not checked
     * @return previous value
     */
    public boolean setIgnoreExternals(boolean ignoreExternals) {
        boolean ret = this.ignoreExternals;
        this.ignoreExternals = ignoreExternals;
        return ret;
    }
}
