/*
 * Copyright 1997-2008 Day Management AG
 * Barfuesserplatz 6, 4001 Basel, Switzerland
 * All Rights Reserved.
 *
 * This software is the confidential and proprietary information of
 * Day Management AG, ("Confidential Information"). You shall not
 * disclose such Confidential Information and shall use it only in
 * accordance with the terms of the license agreement you entered into
 * with Day.
 */
package com.day.cq.search;


import org.osgi.annotation.versioning.ProviderType;

import com.day.cq.search.eval.PredicateEvaluator;
import com.day.cq.search.facets.Bucket;
import com.day.cq.search.result.ResultPage;
import com.day.cq.search.result.SearchResult;

/**
 * A <code>Query</code> represents a JCR repository query that can be created
 * programmatically, with a list of so-called {@link Predicate Predicates}.
 * These act as constraints on the query, meaning they either add predicates to
 * the underlying XPath query or filter the result, and there can be any number
 * of predicates.
 * 
 * <p>
 * Each predicate has a type ({@link Predicate#getType()}) for which a
 * {@link PredicateEvaluator} must exist, that can eg. translate the predicate
 * in question into an XPath predicate. It is possible to register custom
 * {@link PredicateEvaluator PredicateEvaluators} by using the proper OSGi
 * component factory name.
 * 
 * <p>
 * To create queries, use the {@link QueryBuilder} interface. To execute it,
 * simply call {@link #getResult()} to get the {@link SearchResult}, which will
 * contain the hits, metadata about paging and access to facets.
 * 
 * @since 5.2
 */
@ProviderType
public interface Query {

    /**
     * Executes the query and returns the result. Implementations will cache the
     * result, ie. a second call of this method will return the initial result
     * of the query execution; each {@link Query} only runs once. This also
     * means that {@linkplain #addPredicate(PredicateEvaluator) adding} or
     * {@linkplain #removePredicate(PredicateEvaluator) removing} predicates is no
     * longer possible after calling getResult().
     */
    SearchResult getResult();

    /**
     * Returns the list of predicates that define this query. Changing this
     * predicate tree only has an effect before {@link #getResult()} has been
     * called.
     * 
     * @return the root predicate group making up this query
     */
    PredicateGroup getPredicates();

    /**
     * This is an alternate way of providing a custom evaluator for predicates.
     * The normal way is to register them as OSGi component factories (see
     * {@link PredicateEvaluator}), but in some cases it is simpler to inline
     * them, for example to provide a custom facet extraction inside a JSP. A
     * notable difference is that this mechanism is only valid for the current
     * {@link Query}, whereas the OSGi component way of registration makes the
     * evaluator available for all queries.
     * 
     * <p>
     * To register, a <code>type</code> must be specified. This evaluator will
     * be used for all predicates with that type. Evaluators registered this way
     * will have precedence over evaluators provided as OSGi component
     * factories. The type should not start with a "_", as it would be ignored
     * for queries created from requests (see
     * {@link PredicateConverter#createPredicates(java.util.Map)}.
     * 
     * @param type
     *            the predicate type to register the evaluator for
     * @param evaluator
     *            a custom predicate evaluator
     */
    void registerPredicateEvaluator(String type, PredicateEvaluator evaluator);

    /**
     * This will return a new query that includes the given bucket from a
     * previous search. It will simply take the predicate provided by the bucket
     * ({@link Bucket#getPredicate()}) and add it to the new query so that both
     * the existing predicate group <b>and</b> this new predicate are required
     * to match.
     * 
     * @param bucket
     *            a bucket (typically from the facets of this queries search
     *            result)
     * @return a new query that includes the predicate provided by the bucket
     */
    Query refine(Bucket bucket);
    
    // --------------------------------< setter/getters for general params >

    /**
     * Whether the query should return plain nodes or an excerpt. Default is
     * <code>false</code>.
     * 
     * @param excerpt
     *            <code>true</code> if an excerpt should be returned,
     *            <code>false</code> if not
     */
    void setExcerpt(boolean excerpt);

    /**
     * Returns whether the query will return plain nodes or an excerpt. Default
     * is <code>false</code>.
     * 
     * @return <code>true</code> if an excerpt should be returned,
     *         <code>false</code> if not
     */
    boolean getExcerpt();
    
    /**
     * This sets an offset for the actual search results, ie. it will skip the
     * first N (= <code>start</code>) items of the underlying result. By default
     * this is 0, ie. right from the very beginning.
     * 
     * @param start
     *            the offset in the actual search results to start from
     */
    public void setStart(long start);

    /**
     * Returns the offset in the actual search results to start from. See
     * {@link #setStart(long)}.
     * 
     * @return offset in the actual search results to start from
     */
    public long getStart();

    /**
     * Sets the number of hits to include on a {@link ResultPage}. Since only
     * the first page is returned directly and typically fully read by clients,
     * this can also be seen as "limit" for the search result. Further results
     * can be accessed either by retrieving the
     * {@link SearchResult#getNextPage() next result page} or by running a new
     * query and setting the {@link #setStart(long) start} parameter (often also
     * called "offset"). For unlimited results on a single page, use 0. Default
     * value is 10.
     * 
     * @param hitsPerPage
     *            the number of hits to include on a result page (0 for
     *            unlimited results).
     */
    void setHitsPerPage(long hitsPerPage);

    /**
     * Returns the number of hits to include per {@link ResultPage}, ie. the
     * limit. See {@link #setHitsPerPage(long)}.
     * 
     * @return the number of hits to include per result page.
     */
    long getHitsPerPage();
    
    /**
     * Set the amount of results to count to. Default -1 means to count all results.
     * @param guessTotal
     *            the value to count upto.
     */
    void setGuessTotal(long guessTotal);
    
    /**
     * Answer the number of results to count to. To count all results, use -1.
     * @return the number of results to count to.
     */
    long getGuessTotal();
    
    /**
     * Set whether to stop counting results at the current limit. Default is false. 
     * @param guessTotal
     *            whether to stop counting results at the current limit.
     */
    void setGuessTotal(boolean guessTotal);
    
    /**
     * Answer whether the {@link Query} is configured to avoid counting all results.
     * @return true if the query will not count all results. 
     */
    boolean isGuessTotal();
    

}
