/*
 * Copyright 1997-2008 Day Management AG
 * Barfuesserplatz 6, 4001 Basel, Switzerland
 * All Rights Reserved.
 *
 * This software is the confidential and proprietary information of
 * Day Management AG, ("Confidential Information"). You shall not
 * disclose such Confidential Information and shall use it only in
 * accordance with the terms of the license agreement you entered into
 * with Day.
 */
package com.day.cq.search;

import java.util.List;

import javax.jcr.RepositoryException;

import com.day.cq.search.result.ResultPage;
import com.day.cq.search.result.SearchResult;

/**
 * <code>Search</code> implements a simple fulltext search with some additions
 * to the search logic and exposes the query result in a scripting friendly
 * object structure. For more complex queries, see {@link Query} and the
 * {@link QueryBuilder}.
 * 
 * @since 5.2
 */
public interface SimpleSearch {

    /**
     * The prefix for a query searches for similar content.
     */
    String RELATED_PREFIX = "related:";

    // ----------------------------------------------------------< search definition >

    /**
     * Sets a new fulltext query that will be executed.
     * 
     * <p>
     * If this is called after {@link #getResult()} has been called, i.e. the
     * query was executed already, this will lead to the next
     * {@link #getResult()} call executing the new query again.
     * 
     * @param query
     *            the fulltext query.
     */
    public void setQuery(String query);

    /**
     * Returns the query term supplied by the user.
     * 
     * @return the query term supplied by the user.
     */
    String getQuery();

    /**
     * Sets the repository path where to search in. This allows to specify a
     * subtree such as "/content/mysite".
     * 
     * <p>
     * If this is called after {@link #getResult()} has been called, i.e. the
     * query was executed already, this will lead to the next
     * {@link #getResult()} call executing the new query again.
     * 
     * @param searchIn
     *            the location where to search in.
     */
    void setSearchIn(String searchIn);

    /**
     * Returns the repository path where to search in.
     * 
     * @return the location where to search in.
     */
    String getSearchIn();

    /**
     * Sets the number of hits to include on a {@link ResultPage}. Since only
     * the first page is returned directly and typically fully read by clients,
     * this can also be seen as "limit" for the search result. Further results
     * can be accessed either by retrieving the
     * {@link SearchResult#getNextPage() next result page} or by running a new
     * query and setting the {@link #setStart(long) start} parameter (often also
     * called "offset"). For unlimited results on a single page, use 0. Default
     * value is 10.
     * 
     * <p>
     * If this is called after {@link #getResult()} has been called, i.e. the
     * query was executed already, this will lead to the next
     * {@link #getResult()} call executing the new query again.
     * 
     * @param num
     *            the number of hits to include on a result page (0 for
     *            unlimited results)
     */
    void setHitsPerPage(long num);

    /**
     * Returns the number of hits to include per {@link ResultPage}, ie. the
     * limit. See {@link #setHitsPerPage(long)}.
     * 
     * @return the number of hits to include per result page
     */
    long getHitsPerPage();

    /**
     * This sets an offset for the actual search results, ie. it will skip the
     * first N (= <code>start</code>) items of the underlying result iterator.
     * By default this is 0, ie. right from the very beginning.
     * 
     * <p>
     * If this is called after {@link #getResult()} has been called, i.e. the
     * query was executed already, this will lead to the next
     * {@link #getResult()} call executing the new query again.
     * 
     * @param start
     *            the offset in the actual search results to start from
     */
    public void setStart(long start);

    /**
     * Returns the offset in the actual search results to start from. See
     * {@link #setStart(long)}.
     * 
     * @return offset in the actual search results to start from
     */
    public long getStart();

    /**
     * Allows to add custom predicates for the underlying {@link Query}.
     * 
     * <p>
     * If this is called after {@link #getResult()} has been called, i.e. the
     * query was executed already, this will lead to the next
     * {@link #getResult()} call executing the new query again.
     * 
     * @param predicate
     *            a search predicate
     */
    public void addPredicate(Predicate predicate);

    // ----------------------------------------------------------< result >

    /**
     * Returns the search result object. The query is only executed once upon
     * the first call to this method.
     * 
     * @return the query result or <code>null</code> if there is no query
     *         parameter set.
     * @throws RepositoryException
     *             if an exception occurs while executing the query.
     */
    SearchResult getResult() throws RepositoryException;

    /**
     * Returns a object with access to query trend statistics.
     * 
     * @return query trends (popular queries).
     */
    Trends getTrends();

    /**
     * Returns queries that are related to the current one.
     * 
     * @return queries that are related to the current one.
     * @throws RepositoryException
     *             if an error occurs while reading from the repository.
     */
    List<String> getRelatedQueries() throws RepositoryException;

    // ----------------------------------------------------------< config >

    /**
     * Sets the comma-separated list of properties that will be searched for the term given in
     * {@link #setQuery(String)}.
     * 
     * <p>
     * If this is called after {@link #getResult()} has been called, i.e. the
     * query was executed already, this will lead to the next
     * {@link #getResult()} call executing the new query again.
     * 
     * @param properties
     *            comma separated names of the properties that will be searched.
     */
    void setSearchProperties(String properties);

    /**
     * Returns the comma-separated list of properties that will be searched for
     * the term given in {@link #setQuery(String)}.
     * 
     * @return the names of the properties that will be searched.
     */
    String getSearchProperties();
}
