/*
 * Copyright 1997-2008 Day Management AG
 * Barfuesserplatz 6, 4001 Basel, Switzerland
 * All Rights Reserved.
 *
 * This software is the confidential and proprietary information of
 * Day Management AG, ("Confidential Information"). You shall not
 * disclose such Confidential Information and shall use it only in
 * accordance with the terms of the license agreement you entered into
 * with Day.
 */
package com.day.cq.search.eval;

import javax.jcr.Session;
import javax.jcr.query.Row;

import org.osgi.service.component.annotations.Component;
import org.osgi.service.component.annotations.Reference;
import org.osgi.service.component.propertytypes.ServiceVendor;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.day.cq.search.Predicate;
import com.day.cq.search.PredicateGroup;
import com.day.cq.search.Query;
import com.day.cq.search.QueryBuilder;

/**
 * Includes all predicates of a persisted querybuilder query into the current query
 * as a sub {@link PredicateGroupEvaluator}. Note that this will not execute
 * an extra query but extend the current query.
 *
 * <p>
 * Queries can be persisted programmatically using
 * <a href="https://docs.adobe.com/docs/en/aem/6-2/develop/ref/javadoc/com/day/cq/search/QueryBuilder.html#storeQuery(com.day.cq.search.Query,%20java.lang.String,%20boolean,%20Session)">
 * QueryBuilder#storeQuery()</a>. The format can be either a multi-line String
 * property or a nt:file node that contains the query as a text file in Java
 * properties format.
 *
 * <p>
 * Does not support facet extraction for the predicates of the saved query.
 *
 * <h3>Name:</h3>
 * savedquery
 *
 * <h3>Parameters:</h3>
 * <dl>
 * <dt>savedquery</dt>
 * <dd>path to the saved query (String property or nt:file node)</dd>
 * </dl>
 *
 * @since 5.3
 */
@ServiceVendor("Adobe Systems Incorporated")
@Component(factory = "com.day.cq.search.eval.PredicateEvaluator/savedquery")
public class SavedQueryPredicate extends PredicateGroupEvaluator {

    private static final Logger log = LoggerFactory.getLogger(SavedQueryPredicate.class);

    public static final String SAVED_QUERY = "savedquery";

    @Reference
    public QueryBuilder queryBuilder;

    @Override
    public String getXPathExpression(Predicate p, EvaluationContext context) {
        PredicateGroup group = getSavedPredicates(p, context);
        if (group != null) {
            return super.getXPathExpression(group, context);
        }
        // no saved query specified or found => no xpath
        return null;
    }

    @Override
    public boolean includes(Predicate p, Row row, EvaluationContext context) {
        PredicateGroup group = getSavedPredicates(p, context);
        if (group != null) {
            return super.includes(group, row, context);
        }
        // no saved query specified or found => no filtering
        return true;
    }

    @Override
    public boolean canXpath(Predicate p, EvaluationContext context) {
        PredicateGroup group = getSavedPredicates(p, context);
        if (group != null) {
            return super.canXpath(group, context);
        }
        // no saved query specified or found => no xpath
        return false;
    }

    @Override
    public boolean canFilter(Predicate p, EvaluationContext context) {
        PredicateGroup group = getSavedPredicates(p, context);
        if (group != null) {
            return super.canFilter(group, context);
        }
        // no saved query specified or found => no filtering
        return false;
    }

    // ---------------------------------------------------------------< private >

    protected String getEvalContextKey(Predicate p) {
        return p.getPath() + ".savedquery";
    }

    protected PredicateGroup getSavedPredicates(Predicate p, EvaluationContext context) {
        if (!p.hasNonEmptyValue(SAVED_QUERY)) {
            return null;
        }

        String key = getEvalContextKey(p);
        PredicateGroup group = null;
        Object obj = context.get(key);
        if (obj == null) {
            // first time during this evaluation, try to load
            group = loadPredicates(p.get(SAVED_QUERY), context.getSession());
            if (group == null) {
                // set dummy marker to avoid re-loading every time
                context.put(key, new Object());
            } else {
                // persist loaded predicates in evaluation context for later calls
                context.put(key, group);
            }
        } else if (obj instanceof PredicateGroup) {
            group = (PredicateGroup) obj;
        }
        return group;
    }

    protected PredicateGroup loadPredicates(String path, Session session) {
        try {
            // load the predicates from the saved query
            Query query = queryBuilder.loadQuery(path, session);
            if (query == null) {
                // not found
                log.warn("No saved query found at '" + path + "'");
                return null;
            }
            return query.getPredicates();

        } catch (Exception e) {
            log.error("Could not load query from '" + path + "'", e);
            return null;
        }
    }

    /**
     * Bind the QueryBuilder service.
     *
     * @param queryBuilder the QueryBuilder to bind
     * @deprecated 5.13.36 Preserved for maintaining package API baseline.
     */
    @Deprecated
    protected void bindQueryBuilder(QueryBuilder queryBuilder) {
        if (this.queryBuilder == null) {
            this.queryBuilder = queryBuilder;
        }
    }

    /**
     * Unbind the QueryBuilder service.
     *
     * @param queryBuilder the QueryBuilder to unbind
     * @deprecated 5.13.36 Preserved for maintaining package API baseline.
     */
    @Deprecated
    protected void unbindQueryBuilder(QueryBuilder queryBuilder) {
        if (this.queryBuilder == queryBuilder) {
            this.queryBuilder = null;
        }
    }
}
