/*
 * Copyright 1997-2008 Day Management AG
 * Barfuesserplatz 6, 4001 Basel, Switzerland
 * All Rights Reserved.
 *
 * This software is the confidential and proprietary information of
 * Day Management AG, ("Confidential Information"). You shall not
 * disclose such Confidential Information and shall use it only in
 * accordance with the terms of the license agreement you entered into
 * with Day.
 */
package com.day.cq.tagging;

import java.security.AccessControlException;
import java.util.Iterator;
import java.util.List;
import java.util.Locale;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import javax.jcr.Node;
import javax.jcr.Session;

import aQute.bnd.annotation.ProviderType;

import org.apache.sling.api.resource.Resource;
import org.apache.sling.api.resource.ResourceResolver;

import com.day.cq.commons.RangeIterator;


/**
 * <code>TagManager</code> allows for resolving and creating tags by paths and
 * names. See {@link Tag} for a detailed description of tagging concepts,
 * terminology and the structure of tag IDs.
 *
 * <p>
 * This interface is generic, but there is a JCR-based reference implementation
 * which can be obtained by the {@link JcrTagManagerFactory} - all you need is
 * an existing JCR {@link Session} (what tags can be "seen" and which can be
 * created depends on the user of that session):
 *
 * <pre><code>
 * TagManager tagManager = JcrTagManagerFactory.getTagManager(session);
 * </code></pre>
 *
 * In the typical Sling context you can also adapt to a {@link TagManager} from
 * the {@link ResourceResolver}:
 *
 * <pre><code>
 * TagManager tagManager = resourceResolver.adaptTo(TagManager.class);
 * </code></pre>
 */
@ProviderType
public interface TagManager {

    /**
     * Resolves a tag (or namespace) object by a shorthand tag id, such as
     * <code>sky</code> or <code>dam:fruit/apple</code>, or by the absolute
     * path to a tag, such as <code>{base.path}/dam/fruit/apple</code>. Namespaces
     * can be resolved by either using the absolute path to a namespace (one
     * level below the tag base path, eg. normally
     * <code>{base.path}/namespace</code>) or by using the tag id form for
     * namespaces: <code>namespace:</code>
     *
     * @param tagID
     *            a shorthand tag id or an absolute tag path
     * @return a Tag object or <code>null</code> if the tag does not exist
     */
    Tag resolve(String tagID);

    /**
     * Resolves a tag (or namespace) object by a title or path of titles
     * (corresponding to {@link Tag#getTitlePath()}). Note that the system does
     * not ensure unique title paths, but this method should be used before
     * creating a new tag using {@link #createTagByTitle(String)} to avoid basic
     * collisions.
     *
     * @param tagTitlePath
     *            a path that includes titles rather than IDs
     * @return a Tag object (first one found with this title path) or
     *         <code>null</code> if a tag with such a title path does not
     *         exist
     */
    Tag resolveByTitle(String tagTitlePath);

    /**
     * Resolves a tag (or namespace) object by a title or path of titles
     * (corresponding to {@link Tag#getTitlePath()}) in the given locale.
     * 
     * <p>
     * Note that the system does not ensure unique title paths, but this method
     * should be used before creating a new tag using
     * {@link #createTagByTitle(String)} to avoid basic collisions.
     * 
     * @param tagTitlePath
     *            a path that includes titles rather than IDs
     * @param locale
     *            the locale of the titlePath
     * @return a Tag object (first one found with this title path) or
     *         <code>null</code> if a tag with such a title path does not exist
     */
    Tag resolveByTitle(String tagTitlePath, Locale locale);
    
    /**
     * Returns whether the current user (eg. from the underlying JCR session) is
     * allowed to add the specified tag. If the tag already exists,
     * <code>false</code> is returned. Basically it checks if a call
     * to {@link #createTag(String, String, String)} will be successful.
     *
     * @param tagID
     *            a shorthand tag id or an absolute tag path
     * @return <code>true</code> if the current user could create the tag,
     *         <code>false</code> if not or if the tag already exists
     * @throws InvalidTagFormatException
     *             if the parameter <code>tagID</code> is not a valid tag ID
     */
    boolean canCreateTag(String tagID) throws InvalidTagFormatException;

    /**
     * Returns whether the current user (eg. from the underlying JCR session) is
     * allowed to add the tag specified by a title or path of titles
     * (corresponding to {@link Tag#getTitlePath()}). If the tag already
     * exists, <code>false</code> is returned. Basically it checks if a call
     * to {@link #createTagByTitle(String)} will be successful.
     *
     * @param tagTitlePath
     *            a path that includes titles rather than IDs
     * @return <code>true</code> if the current user could create the tag,
     *         <code>false</code> if not. There is no check if the tag exists
     *         already, ie. it could return <code>true</code> if the tag is
     *         already existing.
     * @throws InvalidTagFormatException
     *             if the parameter <code>titlePath</code> references a
     *             non-existing namespace (please note that you cannot create
     *             namespaces via {@link #createTagByTitle(String)})
     */
    boolean canCreateTagByTitle(String tagTitlePath) throws InvalidTagFormatException;

    /**
     * Returns whether the current user (eg. from the underlying JCR session) is
     * allowed to add the tag specified by a title or path of titles
     * (corresponding to {@link Tag#getTitlePath()}). If the tag already exists,
     * <code>false</code> is returned. Basically it checks if a call to
     * {@link #createTagByTitle(String)} will be successful.
     * 
     * <p>
     * This will resolve the parent tag or namespace using the title in the
     * given locale and then set both the default title and the localized title
     * for the new tag.
     * 
     * @param tagTitlePath
     *            a path that includes titles rather than IDs
     * @param locale
     *            the locale of the titlePath
     * @return <code>true</code> if the current user could create the tag,
     *         <code>false</code> if not. There is no check if the tag exists
     *         already, ie. it could return <code>true</code> if the tag is
     *         already existing.
     * @throws InvalidTagFormatException
     *             if the parameter <code>titlePath</code> references a
     *             non-existing namespace (please note that you cannot create
     *             namespaces via {@link #createTagByTitle(String)})
     */
    boolean canCreateTagByTitle(String tagTitlePath, Locale locale) throws InvalidTagFormatException;
    
    /**
     * Creates a new tag (or namespace) by creating it in the tag store, eg.
     * under <code>{base.path}/dam/fruit/apple</code>. If it exists already, the
     * existing tag will be returned, otherwise the object representing the
     * newly created tag. If the current user is not allowed to create a tag, an
     * {@link AccessControlException} will be thrown and no changes will be
     * made.
     *
     * <p>
     * The tag is defined by a shorthand tag id, such as <code>sky</code> or
     * <code>dam:fruit/apple</code>, or by the absolute path to a tag, such as
     * <code>{base.path}/dam/fruit/apple</code>. Namespaces can be created by
     * either using the absolute path to a namespace (one level below the tag
     * base path, eg. normally <code>{base.path}/namespace</code>) or by using
     * the tag id form for namespaces: <code>namespace:</code>
     *
     * <p>This will automatically save the node or session.
     * 
     * @param tagID
     *            a shorthand tag id or an absolute tag path
     * @param title
     *            a title for the tag (can be <code>null</code>)
     * @param description
     *            a longer description for the tag (can be <code>null</code>)
     * @return a Tag object (never <code>null</code>)
     * @throws AccessControlException
     *             if the tag must be created, but the user is not allowed to do
     *             so
     * @throws InvalidTagFormatException
     *             if the parameter <code>tagID</code> is not a valid tag ID
     */
    Tag createTag(String tagID, String title, String description) throws AccessControlException, InvalidTagFormatException;

    /**
     * Creates a new tag (or namespace) by creating it in the tag store, eg.
     * under <code>{base.path}/dam/fruit/apple</code>. If it exists already, the
     * existing tag will be returned, otherwise the object representing the
     * newly created tag. If the current user is not allowed to create a tag, an
     * {@link AccessControlException} will be thrown and no changes will be
     * made.
     *
     * <p>
     * The tag is defined by a shorthand tag id, such as <code>sky</code> or
     * <code>dam:fruit/apple</code>, or by the absolute path to a tag, such as
     * <code>{base.path}/dam/fruit/apple</code>. Namespaces can be created by
     * either using the absolute path to a namespace (one level below the tag
     * base path, eg. normally <code>{base.path}/namespace</code>) or by using
     * the tag id form for namespaces: <code>namespace:</code>
     *
     * @param tagID
     *            a shorthand tag id or an absolute tag path
     * @param title
     *            a title for the tag (can be <code>null</code>)
     * @param description
     *            a longer description for the tag (can be <code>null</code>)
     * @param autoSave whether the session should be automatically saved or not
     * 
     * @return a Tag object (never <code>null</code>)
     * @throws AccessControlException
     *             if the tag must be created, but the user is not allowed to do
     *             so
     * @throws InvalidTagFormatException
     *             if the parameter <code>tagID</code> is not a valid tag ID
     */
    Tag createTag(String tagID, String title, String description, boolean autoSave) throws AccessControlException, InvalidTagFormatException;
    
    /**
     * Creates a new tag (not namespace, can be specified at the beginning, but
     * must exist) from a title or a path of titles (corresponding to
     * {@link Tag#getTitlePath()}).
     *
     * <p>This will automatically save the node or session.
     * 
     * @param titlePath
     *            a path that includes titles rather than IDs
     * @return a Tag object (never <code>null</code>)
     * @throws AccessControlException
     *             if the tag must be created, but the user is not allowed to do
     *             so
     * @throws InvalidTagFormatException
     *             if the parameter <code>titlePath</code> references a
     *             non-existing namespace (please note that this method won't
     *             create namespaces)
     */
    Tag createTagByTitle(String titlePath) throws AccessControlException, InvalidTagFormatException;

    /**
     * Creates a new tag (not namespace, can be specified at the beginning, but
     * must exist) from a title or a path of titles (corresponding to
     * {@link Tag#getTitlePath()}).
     *
     * @param titlePath
     *            a path that includes titles rather than IDs
     * @param autoSave whether the session should be automatically saved or not
     * 
     * @return a Tag object (never <code>null</code>)
     * @throws AccessControlException
     *             if the tag must be created, but the user is not allowed to do
     *             so
     * @throws InvalidTagFormatException
     *             if the parameter <code>titlePath</code> references a
     *             non-existing namespace (please note that this method won't
     *             create namespaces)
     */
    Tag createTagByTitle(String titlePath, boolean autoSave) throws AccessControlException, InvalidTagFormatException;

    /**
     * Creates a new tag (not namespace, can be specified at the beginning, but
     * must exist) from a title or a path of titles (corresponding to
     * {@link Tag#getTitlePath()}). This will resolve the parent tag or
     * namespace using the title in the given locale and then set both the
     * default title and the localized title for the new tag.
     * 
     * <p>
     * This will automatically save the node or session.
     * 
     * @param tagTitlePath
     *            a path that includes titles rather than IDs
     * @param locale
     *            the locale of the titlePath
     * @return a Tag object (never <code>null</code>)
     * @throws AccessControlException
     *             if the tag must be created, but the user is not allowed to do
     *             so
     * @throws InvalidTagFormatException
     *             if the parameter <code>titlePath</code> references a
     *             non-existing namespace (please note that this method won't
     *             create namespaces)
     */
    Tag createTagByTitle(String tagTitlePath, Locale locale) throws AccessControlException, InvalidTagFormatException;
    
    /**
     * Deletes the given tag.
     * 
     * <p>
     * This will automatically save the session.
     * 
     * @param tag
     *            tag to delete
     * @throws AccessControlException
     *             if the user is not allowed to delete the tag
     */
    void deleteTag(Tag tag) throws AccessControlException;

    /**
     * Deletes the given tag.
     * 
     * @param tag
     *            tag to delete
     * @param autoSave
     *            whether the session should be automatically saved or not;
     *            note that if the tag was activated already, it gets automatically
     *            replicated, and as part of this the session will be saved anyway,
     *            thus autoSave=false will only be respected if the tag and its
     *            backlinks have never been activated before
     * @throws AccessControlException
     *             if the user is not allowed to delete the tag
     */
    void deleteTag(Tag tag, boolean autoSave) throws AccessControlException;

    /**
     * Returns all content (Sling {@linkplain Resource Resources}, typically
     * JCR {@linkplain Node Nodes}) tagged with the given tag. In the standard
     * JCR-implementation, these are all nodes with a <code>cq:tags</code>
     * property that contains either the tagID or the full absolute path to that
     * tag. Furthermore, when passing a container tag, such as <code>fruit</code>
     * with eg. sub-tags <code>fruit/apple</code>, <code>fruit/apple/braeburn</code> and
     * <code>fruit/banana</code>, all content tagged with it or one of the
     * sub-tags will be found. Convenience method, see also
     * {@link #find(String, String[])}.
     *
     * @see Tag#find()
     * @see TagManager#find(String, String[])
     *
     * @param tagID
     *            a tag ID or the full path to a tag
     * @return a {@link RangeIterator} containing all found resources (Note:
     *         this is a version of the {@link javax.jcr.RangeIterator} that
     *         supports generics. Returns <code>null</code> if the tag does
     *         not exist
     */
    RangeIterator<Resource> find(String tagID);

    public static class FindResults {
        public Tag[] tags;
        public RangeIterator<Resource> resources;
    }

    /**
     * Searches for all content that is tagged with a tag that contains the
     * given title as tag title. "*" can be used as wildcard in the title. As
     * this method first searches for tags that match the given title, these
     * tags are returned along with the real results in a {@link FindResults}
     * struct-like class.
     * @param title title of tag to find
     * @return a find result
     */
    FindResults findByTitle(String title);

    /**
     * Searches for tags with the given keyword in the title. "*" can be used as wildcard.
     * A locale can be provided to search in a certain localized tag title only.
     * @param keyword
     *            The tag title to be searched
     * @param locale
     *            to search in a certain localized tag title only; use <code>null</code> to search in the default title
     * @return an array of tags found as a result
     */
    Tag[] findTagsByTitle(String keyword, Locale locale);

    /**
     * Returns all content (Sling {@linkplain Resource Resources}, typically
     * JCR {@linkplain Node Nodes}) tagged with all of the given tags, but
     * only content that lies below the given base path.<br>
     * In the standard JCR-implementation, these are all nodes with a
     * <code>cq:tags</code> property that contains either the tagID or the full
     * absolute path to that tag.<br>
     * Furthermore, when passing a container tag, such as <code>fruit</code>
     * with eg. sub-tags <code>fruit/apple</code>,
     * <code>fruit/apple/braeburn</code> and <code>fruit/banana</code>, all
     * content tagged with it or one of the sub-tags will be found.
     *
     * @see Tag#find()
     * @see TagManager#find(String)
     * @see TagManager#find(String, String[], boolean)
     *
     * @param basePath
     *            The starting node of the search
     * @param tagIDs
     *            a list of tag IDs or full paths to tags
     * @return a {@link RangeIterator} containing all found resources (Note:
     *         this is a version of the {@link javax.jcr.RangeIterator} that
     *         supports generics. Returns <code>null</code> if the tag does
     *         not exist
     */
    RangeIterator<Resource> find(String basePath, String[] tagIDs);

    /**
     * Returns all content (Sling {@linkplain Resource Resources}, typically
     * JCR {@linkplain Node Nodes}) tagged with all or one of the given tags,
     * but only content that lies below the given base path.<br>
     * In the standard JCR-implementation, these are all nodes with a
     * <code>cq:tags</code> property that contains either the tagID or the full
     * absolute path to that tag.<br>
     * Furthermore, when passing a container tag, such as <code>fruit</code>
     * with eg. sub-tags <code>fruit/apple</code>,
     * <code>fruit/apple/braeburn</code> and <code>fruit/banana</code>, all
     * content tagged with it or one of the sub-tags will be found.
     *
     * @see Tag#find()
     * @see TagManager#find(String)
     *
     * @param basePath
     *            The starting node of the search
     * @param tagIDs
     *            a list of tag IDs or full paths to tags
     * @param oneMatchIsEnough
     *            if <code>true</code> all the resources are returned that
     *            contain at least one of the given tags.
     * @return a {@link RangeIterator} containing all found resources (Note:
     *         this is a version of the {@link javax.jcr.RangeIterator} that
     *         supports generics. Returns <code>null</code> if the tag does not
     *         exist
     */
    RangeIterator<Resource> find(String basePath, String[] tagIDs, boolean oneMatchIsEnough);

    /**
     * Returns all content (Sling {@linkplain Resource Resources}, typically JCR
     * {@linkplain Node Nodes}) tagged with all or one of the given tags, but
     * only content that lies below the given base path.<br>
     * In the standard JCR-implementation, these are all nodes with a
     * <code>cq:tags</code> property that contains either the tagID or the full
     * absolute path to that tag.<br>
     * Furthermore, when passing a container tag, such as <code>fruit</code>
     * with eg. sub-tags <code>fruit/apple</code>,
     * <code>fruit/apple/braeburn</code> and <code>fruit/banana</code>, all
     * content tagged with it or one of the sub-tags will be found.<br>
     * This method should be used in the case where we need to join more sets of
     * tags. Such as "<code>(a or b) and (c or d)</code>"
     * 
     * @see Tag#find()
     * @see TagManager#find(String)
     * @see TagManager#find(String, String[], boolean)
     * 
     * @param basePath
     *            The starting node of the search
     * @param tagSetIDs
     *            a list of lists of tag IDs or full paths to tags
     * @return a {@link RangeIterator} containing all found resources (Note:
     *         this is a version of the {@link javax.jcr.RangeIterator} that
     *         supports generics. Returns <code>null</code> if the tag does not
     *         exist
     */
    RangeIterator<Resource> find(String basePath, List<String[]> tagSetIDs);

    /**
     * Retrieves all available tag namespaces as array.
     * @return all tag namespaces
     *
     * @see #getNamespacesIter()
     */
    Tag[] getNamespaces();
    
    /**
     * Retrieves all available tag namespaces as iterator.
     * @return an iterator over all tag namespaces
     *
     * @see #getNamespaces()
     */
    Iterator<Tag> getNamespacesIter();

    /**
     * Retrieves the tags set on the given resource. Will return tag
     * objects representing the whole repository (as opposed to
     * {@link #getTagsForSubtree(org.apache.sling.api.resource.Resource, boolean)}).
     *
     * <p>Note that there is no defined order, the result is effectively a set of tags.
     * 
     * @param resource the resource to get the tags from
     * @return all tags for the given resource (in no defined order)
     */
    Tag[] getTags(Resource resource);

    /**
     * Sets tags on the given resource. Please note that this will completely
     * override the previously set tags.
     * 
     * <p>This will automatically save the node or session.
     * 
     * @param resource the resource to set the tags on
     * @param tags the tags to set
     */
    void setTags(Resource resource, Tag[] tags);

    /**
     * Sets tags on the given resource. Please note that this will completely
     * override the previously set tags.
     *
     * @param resource the resource to set the tags on
     * @param tags the tags to set
     * @param autoSave whether the session should be automatically saved or not
     */
    void setTags(Resource resource, Tag[] tags, boolean autoSave);
    
    /**
     * Retrieves the tags set on the given resource and/or all its child resources
     * (aka subtree). The returned tag objects will only represent the subtree,
     * ie. especially the count only applies to the subtree.
     *
     * @param resource the resource to get the tags from
     * @param shallow whether tags only directly on this resource should be used (true)
     *                or the whole subtree is taken into account (false)
     * @return all tags for the given subtree (in no defined order)
     */
    Tag[] getTagsForSubtree(Resource resource, boolean shallow);
    
    /**
     * @deprecated
     * Deprecate in favor of {@link #getResourceResolver()}. This is consistent with recommended
     * api {@link JcrTagManagerFactory#getTagManager(ResourceResolver)}
     *
     * Convenience method that returns the underlying session. Can be used to
     * easily save the session when eg. creating tags with <code>autoSave = false</code>.
     * @return the underlying session
     */
    @Deprecated
    Session getSession();

    /**
     * Convenience method that returns the underlying resource resolver. Can be used to
     * easily save the changes when eg. creating tags with <code>autoSave = false</code>.
     * @return the underlying resource resolver instance
     */
    ResourceResolver getResourceResolver();

    /**
     * Moves a tag.
     * 
     * @param tag
     *            the tag to move
     * @param destination
     *            new location of the tag, given as shorthand tag id or an
     *            absolute tag path
     * @return the new tag at the new location
     * 
     * @throws AccessControlException
     *             if user is not allowed to move the tag
     * @throws InvalidTagFormatException
     *             if the parameter <code>destination</code> is not a valid tag ID
     * @throws TagException
     *             if the tag could not be moved 
     */
    Tag moveTag(Tag tag, String destination) throws AccessControlException, InvalidTagFormatException, TagException;

    /**
     * Merges a tag with another one.
     * 
     * @param tag
     *            the tag to merge into another one (will no longer exist afterwards)
     * @param destination
     *            the tag to merge with (will exist afterwards)
     * 
     * @throws AccessControlException
     *             if user is not allowed to merge the tag
     * @throws TagException
     *             if the tag could not be merged
     */
    void mergeTag(Tag tag, Tag destination) throws AccessControlException, TagException; 
    
    /**
     * Returns the List of language codes support by Tag
     *
     *@return A List of language codes support by Tag (e.i. en,ja,kok,..)
     */
    List<String> getSupportedLanguageCodes();
	
    /**
     * Partial Searches for tags with the given keyword from the title. A locale
     * can be provided to search in a certain localized tag title only. It
     * searches under a specific path only(e.g. {base.path}/NeamSpace/tag1) This
     * API is useful for partial search where full title is not provided to
     * search the tag
     * 
     * @param keyword
     *            The tag title to be searched
     * @param locale
     *            to search in a certain localized tag title only; use <code>null</code> to search in the default title
     * @param tagId
     *            Tag Id of the Tag/NameSpace under which tag to be searched.
     * @return an array of tags found as a result
     */
    Iterable<Tag> findTagsByKeyword(String keyword, Locale locale, String tagId);

}
