/*
 * Copyright 1997-2008 Day Management AG
 * Barfuesserplatz 6, 4001 Basel, Switzerland
 * All Rights Reserved.
 *
 * This software is the confidential and proprietary information of
 * Day Management AG, ("Confidential Information"). You shall not
 * disclose such Confidential Information and shall use it only in
 * accordance with the terms of the license agreement you entered into
 * with Day.
 */

package com.day.cq.wcm.api.components;

import org.apache.sling.commons.json.JSONException;
import org.apache.sling.commons.json.io.JSONWriter;

import java.text.DateFormat;
import java.text.SimpleDateFormat;
import java.util.Locale;

import javax.jcr.Node;
import javax.jcr.NodeIterator;
import javax.jcr.Property;
import javax.jcr.PropertyIterator;
import javax.jcr.PropertyType;
import javax.jcr.RepositoryException;
import javax.jcr.Value;

/**
 * Implements a custom toolbar item that can be initialized with a node reflecting
 * a widget description.
 */
public class JcrToolbarItem implements Toolbar.Item {

    /**
     * the node
     */
    private final Node node;

    /**
     * Creates a custom editing action from the given node
     * @param node the node
     * @throws RepositoryException if a repository error occurs
     */
    public JcrToolbarItem(Node node) throws RepositoryException {
        this.node = node;
    }

    /**
     * {@inheritDoc}
     */
    public void write(JSONWriter writer) throws JSONException {
        try {
            dump(writer, node);
        } catch (RepositoryException e) {
            throw new JSONException("Error while writing JSON.", e);
        }
    }

    /**
     * Dumps the node to the json writer
     * @param out the writer
     * @param node the node
     * @throws JSONException if a JSON error occurs
     * @throws RepositoryException if a repository error occurs
     */
    private void dump(JSONWriter out, Node node)
            throws JSONException, RepositoryException {
        out.object();

        // add node name to ouput
        out.key("name");
        out.value(node.getName());

        PropertyIterator pIter = node.getProperties();
        while (pIter.hasNext()) {
            Property p = pIter.nextProperty();
            if (p.getType() == PropertyType.BINARY) {
                out.key("*" + p.getName());
            } else {
                out.key(p.getName());
            }
            if (p.isMultiple()) {
                out.array();
                Value[] vs = p.getValues();
                for (Value v: vs) {
                    dump(out, v);
                }
                out.endArray();
            } else {
                dump(out, p.getValue());
            }
        }

        NodeIterator nIter = node.getNodes();
        while (nIter.hasNext()) {
            Node child = nIter.nextNode();
            out.key(child.getName());
            dump(out, child);
        }
        out.endObject();
    }

    /**
     * Dumps a value to the json writer
     * @param out the writer
     * @param v the value
     * @throws JSONException if a JSON error occurs
     * @throws RepositoryException if a repository error occurs
     */
    private void dump(JSONWriter out, Value v)
            throws RepositoryException, JSONException {
        if(v.getType() == PropertyType.BINARY) {
            // TODO return the binary size
            out.value(0);
        } else if(v.getType() == PropertyType.DATE) {
            final DateFormat fmt = new SimpleDateFormat("EEE MMM dd yyyy HH:mm:ss 'GMT'Z", Locale.US);
            out.value(fmt.format(v.getDate().getTime()));
        } else {
            out.value(v.getString());
        }
    }

}