/*
 * Copyright 1997-2008 Day Management AG
 * Barfuesserplatz 6, 4001 Basel, Switzerland
 * All Rights Reserved.
 *
 * This software is the confidential and proprietary information of
 * Day Management AG, ("Confidential Information"). You shall not
 * disclose such Confidential Information and shall use it only in
 * accordance with the terms of the license agreement you entered into
 * with Day.
 */
package com.day.cq.wcm.commons;

import java.util.Calendar;

import javax.jcr.AccessDeniedException;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import javax.jcr.Node;
import javax.jcr.RepositoryException;

import com.day.cq.commons.jcr.JcrConstants;
import com.day.cq.wcm.api.NameConstants;
import org.apache.sling.api.resource.ValueMap;
import org.apache.sling.api.servlets.HttpConstants;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * <code>RequestHelper</code>...
 */
public class RequestHelper {

    /**
     * default logger
     */
    private static final Logger log = LoggerFactory.getLogger(RequestHelper.class);

    /**
     * Checks if the request contains a if-last-modified-since header and if the
     * properties have a jcr:lastModified property. if the properties were modified
     * before the header a 304 is sent otherwise the response last modified header
     * is set.
     * @param req the request
     * @param resp the response
     * @param properties the properties
     * @return <code>true</code> if the response was sent
     */
    public static boolean handleIfModifiedSince(HttpServletRequest req,
                                             HttpServletResponse resp,
                                             ValueMap properties) {
        Calendar lastMod = properties.get(JcrConstants.JCR_LASTMODIFIED, Calendar.class);
        if (lastMod == null) {
            lastMod = properties.get(NameConstants.PN_PAGE_LAST_MOD, Calendar.class);
        }
        if (lastMod != null) {
            long modTime = lastMod.getTimeInMillis() / 1000; // seconds
            long ims = req.getDateHeader(HttpConstants.HEADER_IF_MODIFIED_SINCE) / 1000;
            if (modTime <= ims) {
                resp.setStatus(HttpServletResponse.SC_NOT_MODIFIED);
                return true;
            }
            resp.setDateHeader(HttpConstants.HEADER_LAST_MODIFIED, lastMod.getTimeInMillis());
        }
        return false;
    }

    /**
     * Checks if the request contains a if-last-modified-since header and if the
     * node has a jcr:lastModified property. if the properties were modified
     * before the header a 304 is sent otherwise the response last modified header
     * is set. If the give doesn't have the property, the parent node is searched.
     *
     * @param req the request
     * @param resp the response
     * @param node the current node
     * @return <code>true</code> if the response was sent
     */
    public static boolean handleIfModifiedSince(HttpServletRequest req,
                                             HttpServletResponse resp,
                                             Node node) {
        Calendar lastMod = null;
        try {
            Node scan = node;
            while (scan.getDepth() > 0 && lastMod == null) {
                if (scan.hasProperty(JcrConstants.JCR_LASTMODIFIED)) {
                    lastMod = scan.getProperty(JcrConstants.JCR_LASTMODIFIED).getDate();
                } else if (scan.hasProperty(NameConstants.PN_PAGE_LAST_MOD)) {
                    lastMod = scan.getProperty(NameConstants.PN_PAGE_LAST_MOD).getDate();
                } else {
                    scan = scan.getParent();
                }
            }
        } catch (AccessDeniedException e) {
            log.debug("Error while searching for last modified property: " + e);
        } catch (RepositoryException e) {
            log.warn("Error while searching for last modified property: " + e);
        }
        // if no last mod was found above, check if there is a jcr:content/jcr:lastModified
        try {
            if (lastMod == null && node.hasProperty(JcrConstants.JCR_CONTENT + "/" + JcrConstants.JCR_LASTMODIFIED)) {
                lastMod = node.getProperty(JcrConstants.JCR_CONTENT + "/" + JcrConstants.JCR_LASTMODIFIED).getDate();
            }
        } catch (AccessDeniedException e) {
            log.debug("Error while searching for last modified property: " + e);
        } catch (RepositoryException e) {
            log.warn("Error while searching for last modified property: " + e);
        }

        if (lastMod != null) {
            long modTime = lastMod.getTimeInMillis() / 1000; // seconds
            long ims = req.getDateHeader(HttpConstants.HEADER_IF_MODIFIED_SINCE) / 1000;
            if (modTime <= ims) {
                resp.setStatus(HttpServletResponse.SC_NOT_MODIFIED);
                return true;
            }
            resp.setDateHeader(HttpConstants.HEADER_LAST_MODIFIED, lastMod.getTimeInMillis());
        }
        return false;
    }
}