/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * __________________
 *
 *  Copyright 2012 Adobe Systems Incorporated
 *  All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Systems Incorporated and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Systems Incorporated.
 **************************************************************************/
package com.day.cq.wcm.commons;

import com.day.cq.commons.jcr.JcrUtil;
import com.day.cq.wcm.api.Page;
import com.day.cq.wcm.api.PageManager;
import com.day.cq.wcm.api.WCMException;
import com.day.text.Text;
import org.apache.commons.lang.StringUtils;
import org.apache.jackrabbit.JcrConstants;
import org.apache.sling.api.resource.Resource;
import org.apache.sling.api.resource.ResourceResolver;
import org.apache.sling.api.resource.ResourceUtil;

import javax.jcr.Node;
import javax.jcr.RepositoryException;
import javax.jcr.Session;
import java.util.StringTokenizer;

import static com.day.cq.commons.jcr.JcrConstants.JCR_CONTENT;
import static com.day.cq.wcm.commons.Constants.PATH_UGC;

/**
 * Collection of utilities revolving around CQ Social Collaboration features.
 *
 * @since CQ 5.6
 */
public class UGCUtil {

    /**
     * This method maps the given <code>resource</code> to its equivalent path in /usergenerated, e.g. as the
     * relationship a page <i>/content/test</i> to its comment at <i>/content/usergenerated/content/test</i>.
     *
     * @param resource The resource to transpose to the comment's UCG path.
     * @return A <code>String</code> representing the equivalent UCG path.
     */
    public static String resourceToUGCPath(final Resource resource) {
        final StringBuilder path = new StringBuilder(PATH_UGC);

        // append resource path without "/jcr:content/*"
        path.append(getPagePath(resource));

        // append resource name without any selectors / extension
        path.append("/").append(JCR_CONTENT);
        path.append("/").append(getIdFromResource(resource));

        return path.toString();
    }

    /**
     * This method maps the given <code>resource</code> in /content/usergenerated to its equivalent path in the normal
     * content tree, e.g. as the relationship of a comment in <i>//content/usergenerated/content/test</i> to its parent
     * resource at <i>/content/test</i>.
     *
     * @param resource The resource to transpose.
     * @return A <code>String</code> representing the equivalent normal content path.
     */
    public static String UGCToResourcePath(final Resource resource) {
        final String pagePath = StringUtils.substringAfter(resource.getPath(), PATH_UGC);
        return resource.getResourceResolver().map(pagePath);
    }

    /**
     * Returns the given <code>resource</code>'s path without its potential <i>/jcr:content/...</i> suffix, thus
     * representing a page path.
     *
     * @param resource The resource to check.
     * @return A <code>String</code> representing the page path.
     */
    public static String getPagePath(final Resource resource) {
        return StringUtils.substringBefore(resource.getPath(), "/" + JCR_CONTENT);
    }

    /**
     * Gets or creates user generated content node
     *
     * @param resolver The resource resolver.
     * @param pagePath The page path.
     * @return A <code>String</code> representing the UCG path.
     * @throws WCMException If an error occurs while getting or creating user generated content node
     */
    public static String prepareUserGeneratedContent(ResourceResolver resolver, String pagePath) throws WCMException {
        String ugcPagePath = PATH_UGC;
        StringTokenizer pathElems = new StringTokenizer(pagePath, "/");
        while (pathElems.hasMoreTokens()) {
            ugcPagePath += "/" + pathElems.nextToken();
            if (resolver.getResource(ugcPagePath) != null) {
                continue;
            }
            if (ugcPagePath.equals(PATH_UGC + "/content")) {
                // special treatment for root node
                createNode(resolver, ugcPagePath, "sling:Folder");
                save(resolver);
            } else {
                // create pages for parents
                createPage(resolver, ugcPagePath, null, null, null, null);
            }
        }
        return ugcPagePath;
    }

    /**
     * Returns the given <code>resource</code>'s name without selectors and / or file extension.
     *
     * @param resource The resource from which to retrieve its name.
     * @return A <code>String</code> representing the id.
     */
    public static String getIdFromResource(final Resource resource) {
        return StringUtils.substringBefore(ResourceUtil.getName(resource), ".");
    }

    protected static void save(ResourceResolver resolver) throws WCMException, IllegalArgumentException {
        Session session = resolver.adaptTo(Session.class);
        if (session == null) {
            throw new IllegalArgumentException("resolver must be adaptable to session");
        }
        try {
            session.save();
        } catch (RepositoryException re) {
            throw new WCMException("failed to save changes", re);
        }
    }

    protected static Page createPage(ResourceResolver resolver,
                                     String path,
                                     String template,
                                     String resourceTypeProp,
                                     String resourceType,
                                     String title) throws WCMException {
        String parentPath = Text.getRelativeParent(path, 1);
        String name = Text.getName(path);
        return createPage(resolver, parentPath, name, template, resourceTypeProp, resourceType, title);
    }

    protected static Page createPage(ResourceResolver resolver,
                                     String parentPath,
                                     String name,
                                     String template,
                                     String resourceTypeProp,
                                     String resourceType,
                                     String title) throws WCMException {
        Exception e;
        try {
            // ensure parent nodes
            if (resolver.getResource(parentPath) == null) {
                createNode(resolver, parentPath, JcrConstants.NT_UNSTRUCTURED);
                save(resolver);
            }
            if (!JcrUtil.isValidName(name)) {
                name = JcrUtil.createValidName(name, JcrUtil.HYPHEN_LABEL_CHAR_MAPPING);
            }
            Page page = resolver.adaptTo(PageManager.class).create(parentPath, name, template, title);
            if (StringUtils.isNotEmpty(resourceTypeProp)) {
                page.getContentResource().adaptTo(Node.class).setProperty(resourceTypeProp, resourceType);
            }
            return page;
        } catch (RepositoryException re) {
            e = re;
        } catch (WCMException wcme) {
            e = wcme;
        }
        throw new WCMException("failed to create page", e);
    }

    protected static Node createNode(ResourceResolver resolver, String path, String nodeType) throws WCMException {
        if (path.startsWith("/")) {
            path = path.substring(1);
        }
        try {
            Node root = resolver.adaptTo(Session.class).getRootNode();
            // check parents
            StringTokenizer names = new StringTokenizer(Text.getRelativeParent(path, 1), "/");
            while (names.hasMoreTokens()) {
                String name = names.nextToken();
                if (!root.hasNode(name)) {
                    root.addNode(name);
                }
                root = root.getNode(name);
            }
            // add node
            return root.addNode(Text.getName(path), nodeType);
        } catch (Exception e) {
            throw new WCMException("failed to create node", e);
        }
    }

}
