/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * __________________
 *
 *  Copyright 2016 Adobe Systems Incorporated
 *  All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Systems Incorporated and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Systems Incorporated.
 **************************************************************************/

package com.day.cq.wcm.commons.status;

import com.adobe.granite.resourcestatus.ResourceStatus;
import com.adobe.granite.resourcestatus.ResourceStatusProvider;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import java.util.Collections;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;

/**
 * <p>Implementation of a {@link ResourceStatus} for resources that are authored in the Editor
 * (such as pages or templates). {@code EditorResourceStatus}es are retrieved and displayed in a
 * status bar in the Editor.</p>
 *
 * <p>An {@code EditorResourceStatus} consists of the following information:
 * <ul>
 *     <li>A <strong>title</strong> describing the module or feature the status is associated with</li>
 *     <li>A <strong>message</strong> containing the status information</li>
 *     <li>A <strong>priority</strong> (see {@link ResourceStatus#getPriority()})</li>
 *     <li>A <strong>variant</strong> describing the nature of the status (see {@link Variant})</li>
 *     <li>An <strong>icon</strong> (optional, see {@link #getIcon()})</li>
 *     <li>A list of <strong>actions</strong> rendered as clickable links (optional, see {@link Action})</li>
 *     <li>Additional <strong>data</strong> associated with the status (optional)</li>
 * </ul>
 *
 * @see com.adobe.granite.resourcestatus
 * @see ResourceStatus
 */
public class EditorResourceStatus implements ResourceStatus {

    /**
     * Defines the possible variants of statuses, in line with CoralUI.Alert variants.
     */
    public enum Variant {

        /**
         * Indicates that an operation was successful.
         */
        SUCCESS,

        /**
         * Informs the user of non-critical information.
         */
        INFO,

        /**
         * Notifies the user of something important.
         */
        WARNING,

        /**
         * Indicates that an error has occurred.
         */
        ERROR

    }

    private String type;
    private String title;
    private String message;
    private Integer priority;
    private Variant variant;
    private String icon;
    private Map<String, Object> data;
    private List<Action> actions;

    /**
     * Creates a status.
     *
     * @param type the status type (see {@link ResourceStatus#getType()})
     * @param title the title
     * @param message the message
     * @param priority the priority or {@code null}
     * @param variant the variant or {@code null}
     * @param icon the icon or {@code null}
     * @param actions a list of actions or {@code null}
     * @param data additional status data or {@code null}
     * @see EditorResourceStatus
     */
    public EditorResourceStatus(@Nonnull String type, @Nonnull String title, @Nonnull String message,
                                @Nullable Integer priority, @Nullable Variant variant, @Nullable String icon,
                                @Nullable List<Action> actions, @Nullable Map<String, Object> data) {
        this.type = type;
        this.title = title;
        this.message = message;
        this.priority = priority;
        this.variant = variant;
        this.icon = icon;
        this.actions = new LinkedList<Action>();
        if (actions != null) {
            this.actions.addAll(actions);
        }
        this.data = new HashMap<String, Object>();
        if (data != null) {
            this.data.putAll(data);
        }
    }

    /**
     * Private constructor used by {@link Builder} instances.
     *
     * @param builder the builder
     */
    private EditorResourceStatus(Builder builder) {
        this(builder.type, builder.title, builder.message, builder.priority, builder.variant,
                builder.icon, builder.actions, builder.data);
    }

    @Nonnull
    @Override
    public String getType() {
        return this.type;
    }

    @Override
    public int getPriority() {
        // if a custom priority is set
        if (this.priority != null) {
            return this.priority;
        }
        // else return default priorities based on importance of variant:
        // error > warning > info > success
        Variant variant = getVariant();
        if (variant.equals(Variant.ERROR)) {
            return 30000;
        } else if (variant.equals(Variant.WARNING)) {
            return 20000;
        } else if (variant.equals(Variant.INFO)) {
            return 10000;
        }
        return 0;
    }

    @Nullable
    @Override
    public Map<String, Object> getData() {
        // assemble all status data in a new map
        Map<String, Object> data = new HashMap<String, Object>();
        data.putAll(getAdditionalData());
        data.put("title", getTitle());
        data.put("message", getMessage());
        data.put("variant", getVariant().toString().toLowerCase());
        if (getIcon() != null) {
            this.data.put("icon", getIcon());
        }
        List<Action> actions = getActions();
        if (!actions.isEmpty()) {
            // transform list of actions to two arrays
            String[] ids = new String[actions.size()];
            String[] labels = new String[actions.size()];
            int i = 0;
            for (Action action : actions) {
                ids[i] = action.getId();
                labels[i] = action.getLabel();
                i++;
            }
            data.put("actionIds", ids);
            data.put("actionLabels", labels);
        }
        return Collections.unmodifiableMap(data);
    }

    @Nullable
    @Override
    public String getResourceType() {
        // no need for resource type as the status will not be rendered server-side
        return null;
    }

    @Nullable
    @Override
    public String getResourceSuperType() {
        // no need for resource super type as the status will not be rendered server-side
        return null;
    }

    /**
     * Returns the title of this status, which describes the module or feature the status is
     * associated with. In contrast to the the status type (see {@link #getType()}), the title is
     * meant to be displayed in the user interface.
     *
     * @return the status title
     */
    @Nonnull
    public String getTitle() {
        return this.title;
    }

    /**
     * Returns the status message.
     *
     * @return the status message
     */
    public String getMessage() {
        return this.message;
    }

    /**
     * Returns the variant of this status.
     *
     * @return the status variant
     * @see Variant
     */
    @Nonnull
    public Variant getVariant() {
        if (this.variant == null) {
            return Variant.INFO;
        }
        return this.variant;
    }

    /**
     * Returns the name of a CoralUI icon to be displayed in the status. If no icon is set, then a
     * default icon is assumed (depending on the specified {@link Variant}).
     *
     * @return the status icon or {@code null}
     */
    @Nullable
    public String getIcon() {
        return this.icon;
    }

    /**
     * Returns the additional data defined for this status, corresponding to the data that was
     * added using {@link Builder#addData(String, Object)}. In contrast, {@link #getData()} returns
     * a map containing <strong>all</strong> information defining this status (i.e. including title,
     * message etc).
     *
     * @return the additional data of this status
     */
    @Nonnull
    public Map<String, Object> getAdditionalData() {
        return Collections.unmodifiableMap(this.data);
    }

    /**
     * Returns the actions of this status.
     *
     * @return a list of actions
     * @see Action
     */
    @Nonnull
    public List<Action> getActions() {
        return Collections.unmodifiableList(this.actions);
    }

    /**
     * <p>Defines an action for an {@link EditorResourceStatus}. An action is defined by an id and a
     * label, which are used in the user interface to identify and render the action, respectively.
     * </p>
     *
     * <p>Actions are created using {@link Builder#addAction(String, String)}.</p>
     */
    public static class Action {

        private String id;
        private String label;

        private Action(String id, String label) {
            this.id = id;
            this.label = label;
        }

        /**
         * <p>Returns the id of this action. The action id is used in the user interface to identify
         * this action, such that it can be acted upon when triggered.</p>
         *
         * <p>In order to ensure that an action id is unique across different statuses, it should
         * generally be prefixed by the status type (see {@link ResourceStatusProvider#getType()}).</p>
         *
         * @return the action id
         */
        @Nonnull
        public String getId() {
            return this.id;
        }

        /**
         * Returns the label of this action, which is used in the user interface to render the
         * action.
         *
         * @return the action label
         */
        @Nonnull
        public String getLabel() {
            return this.label;
        }

    }

    /**
     * Builder class to build {@link EditorResourceStatus} instances.
     */
    public static class Builder {

        private String type;
        private String title;
        private String message;
        private Integer priority;
        private Variant variant;
        private String icon;
        private Map<String, Object> data;
        private List<Action> actions;

        /**
         * Creates a new {@code Builder}.
         *
         * @param type the status type (see {@link #getType()})
         * @param title the status title (see {@link #getTitle()})
         * @param message the status message (see {@link #getMessage()})
         */
        public Builder(@Nonnull String type, @Nonnull String title, @Nonnull String message) {
            this.type = type;
            this.title = title;
            this.message = message;
            this.data = new HashMap<String, Object>();
            this.actions = new LinkedList<Action>();
        }

        /**
         * Sets the status priority.
         *
         * @param priority the status priority or {@code null} to unset it
         * @return the current {@code Builder}
         * @see #getPriority()
         */
        @Nonnull
        public Builder setPriority(@Nullable Integer priority) {
            this.priority = priority;
            return this;
        }

        /**
         * Sets the status variant.
         *
         * @param variant the status variant or {@code null} to unset it
         * @return the current {@code Builder}
         * @see #getVariant()
         * @see Variant
         */
        @Nonnull
        public Builder setVariant(@Nullable Variant variant) {
            this.variant = variant;
            return this;
        }

        /**
         * Sets the status icon.
         *
         * @param icon the status icon or {@code null} to unset it
         * @return the current {@code Builder}
         * @see #getIcon()
         */
        @Nonnull
        public Builder setIcon(@Nullable String icon) {
            this.icon = icon;
            return this;
        }

        /**
         * Associates additional data with this status. This data is available as properties when
         * retrieving status resources (see {@link ResourceStatus}).
         *
         * @param key the key
         * @param value the value
         * @return this
         */
        public Builder addData(String key, Object value) {
            this.data.put(key, value);
            return this;
        }

        /**
         * <p>Associates an action with this status. </p>
         *
         * @param id the action id (see {@link Action#getId()})
         * @param label the action label (see {@link Action#getLabel()})
         * @return this
         */
        public Builder addAction(String id, String label) {
            this.actions.add(new Action(id, label));
            return this;
        }

        /**
         * Builds and returns the {@link EditorResourceStatus} instance.
         *
         * @return the {@link EditorResourceStatus} instance
         */
        public EditorResourceStatus build() {
            return new EditorResourceStatus(this);
        }

    }

}
