/*
 * Copyright 1997-2010 Day Management AG
 * Barfuesserplatz 6, 4001 Basel, Switzerland
 * All Rights Reserved.
 *
 * This software is the confidential and proprietary information of
 * Day Management AG, ("Confidential Information"). You shall not
 * disclose such Confidential Information and shall use it only in
 * accordance with the terms of the license agreement you entered into
 * with Day.
 */
package com.day.cq.wcm.core.stats;

import java.net.MalformedURLException;
import java.net.URI;
import java.net.URISyntaxException;
import java.net.URL;
import java.util.Dictionary;
import java.util.Iterator;

import javax.jcr.Node;
import javax.jcr.RepositoryException;
import javax.jcr.Session;

import org.apache.felix.scr.annotations.Component;
import org.apache.felix.scr.annotations.Property;
import org.apache.felix.scr.annotations.Reference;
import org.apache.felix.scr.annotations.ReferencePolicy;
import org.apache.felix.scr.annotations.Service;
import org.osgi.service.component.ComponentContext;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.day.cq.statistics.StatisticsService;
import com.day.cq.wcm.api.Page;
import com.day.cq.wcm.api.WCMException;
import com.day.cq.wcm.api.WCMMode;

/**
 * Little utility for Page-Statistics. Allows to configure the
 * {@link java.net.URL URL} to be called for tracking page-views. Contains
 * convenience for the running of the impression statistics. Is a hook to add
 * possible access to remote reports.
 *
 * @see PageViewStatistics
 */
@Component(metatype=true)
@Service
public class PageViewStatisticsImpl implements PageViewStatistics {
    private final static Logger log = LoggerFactory
            .getLogger(PageViewStatisticsImpl.class);

    @Property
    private static final String TRACKING_URL_PROPERTY = "pageviewstatistics.trackingurl";

    @Property(label = "Tracking script enabled",
            description = "Enable or disable the inclusion of the tracking script on the pages. No page tracking is available when disabled ")
    private static final String TRACKING_SCRIPT_ENABLED = "pageviewstatistics.trackingscript.enabled";

    @Reference(policy=ReferencePolicy.STATIC)
    private StatisticsService statistics; // set via scr

    private boolean isLocal = true;

    private String trackingUrl;

    private boolean trackingScriptEnabled = false;

    private String dataPath;

    // -----------------------------------------------< PageViewStatistics >----

    /**
     * @see com.day.cq.wcm.core.stats.PageViewStatistics#getTrackingURL()
     */
    public URL getTrackingURL() {
        if (!trackingScriptEnabled) {
            return null;
        }
        return getURL(trackingUrl);
    }

    /**
     * @see com.day.cq.wcm.core.stats.PageViewStatistics#getTrackingURL()
     */
    public URI getTrackingURI() {
        if (!trackingScriptEnabled) {
            return null;
        }
    	return getURI(trackingUrl);
    }

    /**
     * @see com.day.cq.wcm.core.stats.PageViewStatistics#report(com.day.cq.wcm.api.Page)
     */
    public Object[] report(Page page) throws WCMException {
        if (isLocal) {
            return reportLocal(page);
        } else {
            log.debug("Remote reports not implemented");
        }
        return null;
    }

    // -----------------------------------------------< SCR Integration >-------

    protected void activate(ComponentContext context) {
        Dictionary<?, ?> props = context.getProperties();

        trackingUrl = (String) props.get(TRACKING_URL_PROPERTY);
        trackingScriptEnabled = Boolean.parseBoolean((String) props.get(TRACKING_SCRIPT_ENABLED));

        dataPath = statistics.getPath() + "/pages";
    }

    // -----------------------------------------------< Internal >--------------

    /**
     * Run a report for the given page
     *
     * @param page
     *            to report
     * @return report or <code>null</code> if none at all
     * @throws WCMException
     *             in case of error running report
     */
    private Object[] reportLocal(Page page) throws WCMException {
        try {
            PageViewReport report = new PageViewReport(dataPath, page,
                    WCMMode.PREVIEW);
            report.setPeriod(1);
            Node node = page.adaptTo(Node.class);
            Session s = node != null ? node.getSession() :
                    page.getContentResource().getResourceResolver().adaptTo(Session.class);
            Iterator<?> result = statistics.runReport(s, report);
            if (result.hasNext()) {
                return (Object[]) result.next();
            }
            return null;
        } catch (RepositoryException e) {
            throw new WCMException(e);
        }
    }

    private URL getURL(String urlProperty) {
        if (urlProperty != null) {
            try {
                return new URL(urlProperty);
            } catch (MalformedURLException e) {
                log.error("Configuration contained URL that is not vald{}: {}",
                        urlProperty, e);
            }
        }
        return null;
    }

    private URI getURI(String urlProperty) {
    	if (urlProperty != null) {
            try {
                return new URI(urlProperty);
            } catch (URISyntaxException e) {
                log.error("Configuration contained URI that is not valid{}: {}",
                		urlProperty, e);
            }
        }
        return null;
    }
}
