/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * ___________________
 *
 *  Copyright 2012 Adobe Systems Incorporated
 *  All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Systems Incorporated and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Systems Incorporated.
 **************************************************************************/
package com.day.cq.wcm.designimporter.parser;

import java.util.List;

import com.day.cq.wcm.designimporter.UnsupportedTagContentException;
import com.day.cq.wcm.designimporter.DesignImportException;
import com.day.cq.wcm.designimporter.DesignImporterContext;
import com.day.cq.wcm.designimporter.ErrorCodes;
import com.day.cq.wcm.designimporter.MissingCanvasException;
import org.xml.sax.Attributes;
import org.xml.sax.ContentHandler;
import org.xml.sax.Locator;
import org.xml.sax.SAXException;

import com.day.cq.dam.indd.PageBuilder;
import com.day.cq.dam.indd.PageComponent;
import com.day.cq.wcm.designimporter.api.EntryTagHandler;
import com.day.cq.wcm.designimporter.api.HTMLContentProvider;
import com.day.cq.wcm.designimporter.api.PageComponentProvider;
import com.day.cq.wcm.designimporter.api.TagHandler;
import com.day.cq.wcm.designimporter.api.TagHandlerProvider;
import com.day.cq.wcm.designimporter.parser.taghandlers.CanvasComponentTagHandler;
import com.day.cq.wcm.designimporter.parser.taghandlers.HeadTagHandler;

/**
 * The SAX content handler for handling the HTML SAX events raised by the HTML parser.
 */
public class HTMLContentHandler implements ContentHandler {
    /**
     * The importer context associated with the import request.
     */
    protected DesignImporterContext designImporterContext;

    private boolean canvasFound;

    private String canvasResourceType;

    private int counter;

    private HTMLContent headHtmlContent = new HTMLContent();

    private HTMLContent bodyHtmlContent = new HTMLContent();

    private String language;

    private PageBuilder pageBuilder;

    private List<PageComponent> pageComponents;

    private TagHandler tagHandler;

    private TagHandlerProvider tagHandlerProvider;

    public void characters(char[] ch, int start, int length) throws SAXException {
        try {
            // Delegate if there is a tag handler currently handling.
            if (tagHandler != null) tagHandler.characters(ch, start, length);
        } catch (DesignImportException e) {
            throw new SAXException(e);
        }
    }

    public void endDocument() throws SAXException {
        if (!canvasFound) throw new SAXException(new MissingCanvasException());
    }

    public void endElement(String uri, String localName, String qName) throws SAXException {
        try {
            if (tagHandler instanceof EntryTagHandler) {
                if (--counter == 0) { // termination condition
                    tagHandler.endHandling(uri, localName, qName);
                    extractComponentsAndContent(tagHandler);
                    tagHandler = null;
                } else {
                    tagHandler.endElement(uri, localName, qName);
                }
            }
        } catch (DesignImportException e) {
            throw new SAXException(e);
        }
    }

    public void endPrefixMapping(String prefix) throws SAXException {
    }

    public HTMLContent getBodyHtmlContent() {
        return bodyHtmlContent;
    }

    /**
     * {@link PageComponent}s are extracted from the input HTML stream on the basis of the parameterized component divs and the
     * {@link TagHandler}s registered with the {@link TagHandlerProvider}
     * 
     * @return The list of {@link PageComponent}s extracted from the input HTML stream.
     */
    public List<PageComponent> getGeneratedComponents() {
        return pageComponents;
    }

    public HTMLContent getHeadHtmlContent() {
        return headHtmlContent;
    }

    public String getLanguage() {
        return language;
    }

    public void ignorableWhitespace(char[] ch, int start, int length) throws SAXException {
    }

    public void processingInstruction(String target, String data) throws SAXException {
    }

    /**
     * Sets the unique canvas resourceType
     * 
     * @param canvasResourceType
     */
    public void setCanvasResourceType(String canvasResourceType) {
        this.canvasResourceType = canvasResourceType;
    }

    public void setDocumentLocator(Locator locator) {
    }

    /**
     * Sets the design importer context
     * 
     * @param designImporterContext The {@link DesignImporterContext} context object
     */
    public void setDesignImporterContext(DesignImporterContext designImporterContext) {
        this.designImporterContext = designImporterContext;
    }

    /**
     * Sets the page builder for building page components.
     * 
     * @param pageBuilder The {@link PageBuilder} object for building {@link PageComponent}s
     */
    public void setPageBuilder(PageBuilder pageBuilder) {
        this.pageBuilder = pageBuilder;
    }

    /**
     * Sets the {@link TagHandlerProvider}
     *
     * @param tagHandlerProvider The {@link TagHandlerProvider} instance
     */
    public void setTagHandlerProvider(TagHandlerProvider tagHandlerProvider) {
        this.tagHandlerProvider = tagHandlerProvider;
    }

    public void skippedEntity(String name) throws SAXException {
    }

    public void startDocument() throws SAXException {
        canvasFound = false;
    }

    public void startElement(String uri, String localName, String qName, Attributes atts) throws SAXException {
        if ("html".equalsIgnoreCase(localName)) {
            if (atts != null) this.language = atts.getValue("lang");
        }
        try {
            // If there is a tag handler currently handling, delegate to it.
            if (tagHandler != null) {
                tagHandler.startElement(uri, localName, qName, atts);
                counter++;
            } else { // Otherwise, try to create a new tag handler for delegation purposes.
                tagHandler = tagHandlerProvider.createTagHandler(localName, atts);
                if (tagHandler instanceof EntryTagHandler) {
                    tagHandler.setDesignImporterContext(designImporterContext);
                    tagHandler.setTagHandlerProvider(tagHandlerProvider);
                    if (tagHandler instanceof CanvasComponentTagHandler) {
                        if (canvasFound) {
                            // Only one canvas is supported.
                            throw new UnsupportedTagContentException(
                                    ErrorCodes.MULTIPLE_DIV_TAGS);
                        }
                        canvasFound = true;
                        ((CanvasComponentTagHandler) tagHandler).setPageBuilder(pageBuilder);
                        ((CanvasComponentTagHandler) tagHandler).setResourceType(canvasResourceType);
                    }

                    tagHandler.beginHandling(uri, localName, qName, atts);
                    counter = 1;
                } else {
                    tagHandler = null;
                }
            }
        } catch (DesignImportException e) {
            throw new SAXException(e);
        }
    }

    public void startPrefixMapping(String prefix, String uri) throws SAXException {
    }

    private void extractComponentsAndContent(TagHandler tagHandler) {

        if (tagHandler instanceof PageComponentProvider) {
            pageComponents = ((PageComponentProvider) tagHandler).getPageComponents();
        }

        if (tagHandler instanceof HTMLContentProvider) {
            for (HTMLContentType contentType : HTMLContentType.values()) {
                if (((HTMLContentProvider) tagHandler).supportsContent(contentType)) {
                    Object content = ((HTMLContentProvider) tagHandler).getContent(contentType);

                    if (content != null) {
                        if (tagHandler instanceof HeadTagHandler)
                            headHtmlContent.add(contentType, content);
                        else
                            bodyHtmlContent.add(contentType, content);
                    }
                }
            }
        }

    }

}
