/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * ___________________
 *
 *  Copyright 2012 Adobe Systems Incorporated
 *  All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Systems Incorporated and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Systems Incorporated.
 **************************************************************************/
package com.day.cq.wcm.designimporter.parser.taghandlers;

import java.util.HashMap;
import java.util.Map;

import javax.jcr.RepositoryException;

import com.day.cq.wcm.designimporter.ErrorCodes;
import org.apache.sling.api.resource.ValueMap;
import org.apache.sling.api.wrappers.ValueMapDecorator;
import org.ccil.cowan.tagsoup.AttributesImpl;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.xml.sax.Attributes;

import com.day.cq.commons.DownloadResource;
import com.day.cq.dam.indd.PageComponent;
import com.day.cq.wcm.designimporter.DesignImportException;
import com.day.cq.wcm.designimporter.UnsupportedTagContentException;
import com.day.cq.wcm.designimporter.parser.HTMLContentType;
import com.day.cq.wcm.designimporter.util.TagUtils;

/**
 * The image component tag handler is used to translate the image component div(or an equivalent markup)
 * into a CQ image component.
 *
 * <p>
 * This tag handler is registered against the markup of type < div|span|img data-cq-component="image" >
 * (Read as a div, span, img tag with the attribute data-cq-component equal to "image"). That implies
 * this tag handler will come into action whenever the design importer framework
 * encounters an HTML tag matching the rule this handler is defined against.
 * </p>
 *
 * @see com.day.cq.wcm.designimporter.parser.taghandlers.factory.ImageComponentTagHandlerFactory
 */
public class ImageComponentTagHandler extends AbstractTagHandler {

    private Logger logger = LoggerFactory.getLogger(ImageComponentTagHandler.class);

    private String imageFileReference;

    private String alt;

    private String title;

    private String height;

    private String width;

    private String usemap;

    private String suffix;

    private String resourceType;

    @Override
    public void beginHandling(String uri, String localName, String qName, Attributes atts) throws DesignImportException {
        super.beginHandling(uri, localName, qName, atts);

        if ("img".equalsIgnoreCase(localName)) {
            AttributesImpl modAttibutes = new AttributesImpl(atts);
            int index = atts.getIndex("data-cq-component");
            if (index > -1) modAttibutes.removeAttribute(index);

            startElement(uri, localName, qName, modAttibutes);
        }
    }

    @Override
    public void startElement(String uri, String localName, String qName, Attributes atts) throws DesignImportException {
        if (imageFileReference != null)
            throw new UnsupportedTagContentException(ErrorCodes.SINGLE_IMG_IMAGE);

        super.startElement(uri, localName, qName, atts);

        if (!(delegate instanceof ImgTagHandler))
            throw new UnsupportedTagContentException(ErrorCodes.SINGLE_IMG_IMAGE);
    }

    @Override
    public void characters(char[] ch, int start, int length) throws DesignImportException {
        super.characters(ch, start, length);
        if (delegate == null && new String(ch, start, length).trim().length() > 0) // Only WS characters allowed. Any text is not tolerable.
            throw new UnsupportedTagContentException(
                    ErrorCodes.FLOATING_TEXT_SINGLE_IMG_IMAGE);
    }

    @Override
    public void endElement(String uri, String localName, String qName) throws DesignImportException {
        if (delegate instanceof ImgTagHandler) {
            if (((ImgTagHandler) delegate).isRawSrcAbsolute())
                throw new UnsupportedTagContentException(
                        ErrorCodes.ABSOLUTE_URL_NOT_SUPPORTED);

            imageFileReference = getDesignPath() + "/" + ((ImgTagHandler) delegate).getRawSrc();

            alt = ((ImgTagHandler) delegate).getAlt();
            title = ((ImgTagHandler) delegate).getTitle();
            height = ((ImgTagHandler) delegate).getHeight();
            width = ((ImgTagHandler) delegate).getWidth();
            usemap = ((ImgTagHandler) delegate).getUsemap();
        }
        super.endElement(uri, localName, qName);
    }

    private String getDesignPath() {
        try {
            return designImporterContext.designNode.getPath();
        } catch (RepositoryException e) {
            logger.error("A repository exception occured while trying to obtain the design path from the page context", e);
        }
        return "";
    }

    @Override
    public void endHandling(String uri, String localName, String qName) throws DesignImportException {
        super.endHandling(uri, localName, qName);
        if ("img".equalsIgnoreCase(localName)) {
            endElement(uri, localName, qName);
        }

        ValueMap properties = new ValueMapDecorator(getPropertyMap());
        suffix = designImporterContext.componentSuffixGenerator.getSuffix("image");
        PageComponent imageComponent = pageBuilder.createComponent(resourceType, properties, getNameHint());
        getPageComponents().add(imageComponent);
    }

    @Override
    public boolean supportsContent(HTMLContentType htmlContentType) {
        if (htmlContentType == HTMLContentType.MARKUP)
            return true;
        else
            return false;
    }

    private String getNameHint() {
        return "image" + suffix;
    }

    @Override
    public Object getContent(HTMLContentType htmlContentType) {
        if (htmlContentType == HTMLContentType.MARKUP) {
            String cqIncludeJspTag = "<sling:include path=" + "\"" + getNameHint() + "\"" + "/>";
            return componentDivStartTag + cqIncludeJspTag + TagUtils.getMatchingEndTag(componentDivStartTag);
        }
        return null;
    }

    private Map<String, Object> getPropertyMap() {
        Map<String, Object> base = new HashMap<String, Object>();
        base.put(DownloadResource.PN_REFERENCE, imageFileReference);

        if (alt != null) base.put("alt", alt);
        if (title != null) base.put("jcr:title", title);
        if (width != null) base.put("width", width);
        if (height != null) base.put("height", height);
        if (usemap != null) {
            if (usemap.startsWith("#")) {
                base.put("imageMap", usemap.substring(1));
            } else {
                base.put("imageMap", usemap);
            }
        }

        return base;
    }

    public void setResourceType(String resourceType) {
        this.resourceType = resourceType;
    }

}
