/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * ___________________
 *
 *  Copyright 2012 Adobe Systems Incorporated
 *  All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Systems Incorporated and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Systems Incorporated.
 **************************************************************************/
package com.day.cq.wcm.designimporter.parser.taghandlers;

import com.adobe.granite.xss.XSSAPI;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.xml.sax.Attributes;

import javax.jcr.RepositoryException;
import java.net.MalformedURLException;
import java.net.URL;

/**
 * This tag handler is used to transform the src attribute within an img tag.
 *
 * <p>
 * Since the images should continue to be referable, their src attribute values are prepended with the current design path. Note that design
 * path is where the images are dumped during the import process.
 * </p>
 *
 * @see com.day.cq.wcm.designimporter.parser.taghandlers.factory.ImgTagHandlerFactory
 */
public class ImgTagHandler extends DefaultTagHandler {

    private String title;

    private String alt;

    private String width;

    private String height;

    private String usemap;

    public XSSAPI xssAPI;

    public String getTitle() {
        return title;
    }

    public String getAlt() {
        return alt;
    }

    public String getWidth() {
        return width;
    }

    public String getHeight() {
        return height;
    }

    public String getUsemap() {
        return usemap;
    }

    private String rawSrc;

    public String getRawSrc() {
        return rawSrc;
    }

    private String normalizedSrc;

    public boolean isRawSrcAbsolute() {
        return isAbsoluteUrl(rawSrc);
    }

    private boolean normalizationRequired = true;

    private String normalizationPrefix;

    public void setNormalizationPrefix(String normalizationPrefix) {
        this.normalizationPrefix = normalizationPrefix;
    }

    public boolean isNormalizationRequired() {
        return normalizationRequired;
    }

    public void setNormalizationRequired(boolean normalizationRequired) {
        this.normalizationRequired = normalizationRequired;
    }

    private Logger logger = LoggerFactory.getLogger(ImgTagHandler.class);

    @Override
    public void beginHandling(String uri, String localName, String qName, Attributes atts) {
        super.beginHandling(uri, localName, qName, atts);

        // This tag handler is required to prepend the design path of the
        // current page ahead of the relative image url. Since the images are
        // all being dumped into the design path directory and stay no more
        // relative to the page HTML eventually produced.

        width = atts.getValue("width");
        height = atts.getValue("height");
        alt = atts.getValue("alt");
        title = atts.getValue("title");
        usemap = atts.getValue("usemap");
        normalizedSrc = rawSrc = atts.getValue("src");

        if(normalizationPrefix == null) {
            normalizationPrefix = getDesignPath() + "/";
        }

    }

    @Override
    protected String getStartTag(String uri, String localName, String qName, Attributes atts) {

        if (rawSrc != null) {
            if (!isRawSrcAbsolute() && isNormalizationRequired()) {
                normalizedSrc = normalizationPrefix + rawSrc;
            }
        }

        String startTag = "<" + localName.toLowerCase();
        String att = "";
        for (int i = 0; i < atts.getLength(); i++) {
            String attrName = atts.getLocalName(i);
            String attrValue = atts.getValue(i);
            if ("src".equalsIgnoreCase(attrName)) {
                att += " " + attrName + "=\"" + normalizedSrc + "\"";
            } else {
                att += " " + attrName + "=\"" + attrValue + "\"";
            }
        }
        startTag += att;
        startTag += "/>";

        // Check if the image was existent in the archive. If not, log a warning.
        try {
            if (rawSrc != null && !designImporterContext.designNode.hasNode(rawSrc)) {
                designImporterContext.importWarnings.add("Could not locate the referenced image '" + rawSrc + "' in the design package");
            }
        } catch (RepositoryException e) {
        }

        return startTag;
    }

    protected boolean isAbsoluteUrl(String url) {
        try {
            new URL(url);
            return true;
        } catch (MalformedURLException e) {
            return false;
        }
    }

    private String getDesignPath() {
        try {
            String designPath = designImporterContext.designNode.getPath();
            if(xssAPI != null) {
                // getValidHref does namespace mangling, which is needed for the
                // AntiSamy policy not to filter tags containing a colon (this
                // happened for images nested in text components, see CQ5-27685)
                designPath = xssAPI.getValidHref(designPath);
            }
            return designPath;
        } catch (RepositoryException e) {
            logger.error("Error occurred while trying to access the path of the design node", e);
        }
        return "";
    }

    public void setXssAPI(XSSAPI xssAPI) {
        this.xssAPI = xssAPI;
    }

}
