/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * ___________________
 *
 *  Copyright 2012 Adobe Systems Incorporated
 *  All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Systems Incorporated and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Systems Incorporated.
 **************************************************************************/
package com.day.cq.wcm.designimporter.parser.taghandlers;

import java.net.MalformedURLException;
import java.net.URL;
import java.util.Arrays;

import javax.jcr.RepositoryException;

import com.day.cq.wcm.designimporter.ErrorCodes;
import org.xml.sax.Attributes;

import com.day.cq.wcm.designimporter.DesignImporterContext;
import com.day.cq.wcm.designimporter.DesignImportException;
import com.day.cq.wcm.designimporter.UnsupportedTagContentException;
import com.day.cq.wcm.designimporter.api.EntryTagHandler;
import com.day.cq.wcm.designimporter.api.HTMLContentProvider;
import com.day.cq.wcm.designimporter.api.TagHandler;
import com.day.cq.wcm.designimporter.api.TagHandlerProvider;
import com.day.cq.wcm.designimporter.parser.HTMLContentType;

/**
 * The link tag handler is used to extract the name of the stylesheets referenced
 * within the input HTML document via the < link > element
 *
 * <p>
 * This tag handler is registered against the markup of type < link rel="stylesheet" >. That implies
 * this tag handler will come into action whenever the design importer framework
 * encounters an HTML tag matching the rule this handler is defined against.
 * </p>
 *
 * <p>
 * Note: This tag handler does not handle inline script tags. {@see InlineScriptTagHandler}
 * </p>
 *
 * @see com.day.cq.wcm.designimporter.parser.taghandlers.factory.LinkTagHandlerFactory
 */
public class LinkTagHandler implements TagHandler, EntryTagHandler, HTMLContentProvider {


    private String cssPath;

    private Boolean isAbsoluteStylesheet = false;

    private DesignImporterContext designImporterContext;

    public void beginHandling(String uri, String localName, String qName, Attributes atts) {

        String relation = atts.getValue("rel");
        if ("stylesheet".equalsIgnoreCase(relation)) {
            if(isSupportedMedia(atts)) {
                if ((cssPath = atts.getValue("href")) != null) {

                    if (isValidURL(cssPath)) {
                        isAbsoluteStylesheet = true;
                    } else {
                        cssPath = cssPath.replaceAll("([^?]*)?.*", "$1"); // Sanitize the css url
                        // Check if the css was existent in the archive. If not, log a warning.
                        try {
                            if (!designImporterContext.designNode.hasNode(cssPath)) {
                                designImporterContext.importWarnings.add("Could not locate the referenced stylesheet '" + cssPath
                                        + "' in the design package");
                            }
                        } catch (RepositoryException e) {
                        }
                    }
                }
            }
        }
    }

    private boolean isSupportedMedia(Attributes atts) {
        String media = atts.getValue("media");

        if(media == null || media.length() == 0)
            return true;
        return media.matches("(all|screen)");
    }

    private boolean isValidURL(String cssPath) {
        try {
            new URL(cssPath);
            return true;
        } catch (MalformedURLException e) {
            return false;
        }
    }

    public void characters(char[] ch, int start, int length) throws DesignImportException {
        if (new String(ch, start, length).trim().length() > 0)
            throw new UnsupportedTagContentException(ErrorCodes.FREE_FLOATING_TEXT_LINK_TAG);
    }

    public void endElement(String uri, String localName, String qName) throws DesignImportException {
        // Invalid. Link tag must not contain any nested tags.
        throw new UnsupportedTagContentException();
    }

    public void endHandling(String uri, String localName, String qName) {

    }

    public Object getContent(HTMLContentType htmlContentType) {
        if (htmlContentType == HTMLContentType.STYLESHEET_INCLUDE) {
            if (cssPath != null && cssPath.trim().length() > 0) return Arrays.asList(new String[] { cssPath });
        } else if (htmlContentType == HTMLContentType.MARKUP) return "<link rel='stylesheet' type='text/css' href='" + cssPath + "' />";
        return null;
    }

    public void setDesignImporterContext(DesignImporterContext designImporterContext) {
        this.designImporterContext = designImporterContext;
    }

    public void setTagHandlerProvider(TagHandlerProvider tagHandlerProvider) {
    }

    public void startElement(String uri, String localName, String qName, Attributes atts) throws DesignImportException {
        // Invalid. Link tag must not contain any nested tags.
        throw new UnsupportedTagContentException();
    }

    public boolean supportsContent(HTMLContentType htmlContentType) {
        if (htmlContentType == HTMLContentType.STYLESHEET_INCLUDE)
            return !isAbsoluteStylesheet;
        else if (htmlContentType == HTMLContentType.MARKUP)
            return isAbsoluteStylesheet;
        else
            return false;
    }
}