/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * __________________
 *
 *  Copyright 2016 Adobe Systems Incorporated
 *  All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Systems Incorporated and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Systems Incorporated.
 **************************************************************************/
package com.day.cq.wcm.foundation.utils;

import com.adobe.granite.xss.XSSAPI;
import org.apache.commons.lang.StringUtils;
import org.apache.sling.api.SlingHttpServletRequest;
import org.apache.sling.api.SlingHttpServletResponse;
import org.apache.sling.commons.json.JSONException;
import org.apache.sling.commons.json.JSONObject;
import org.apache.sling.commons.json.sling.JsonObjectCreator;

import java.io.IOException;
import java.util.Iterator;

/**
 * Rich Text Editor Utilities
 */
@SuppressWarnings("unused")
public class RTEUtils {
    private final static String PROPERTY_TEXT_IS_RICH = "textIsRich";
    private final static String INFINITY = "infinity";

    /**
     * Create a JSON response containing XSS filtered JSON of a resource.
     * It will recursively filter all properties of name 'propertyName' if the respective node has 'textIsRich' set to "true",
     * The maximum recursion depth (or INFINITY) is read from the selector
     *
     * @param request      The Sling request
     * @param propertyName The name of the property to filter
     * @param xssApi       A reference to the XSSAPI to use
     * @param filterXSS    Whether or not to enable the XSS filter
     * @throws JSONException
     * @throws IOException
     */
    @SuppressWarnings("unused")
    public static void createFilteredJSONResponse(SlingHttpServletRequest request, SlingHttpServletResponse response,
                              String propertyName, XSSAPI xssApi, boolean filterXSS) throws JSONException, IOException {
        response.setContentType("application/json");
        response.setCharacterEncoding("utf-8");
        JSONObject jsonObj = JsonObjectCreator.create(request.getResource(), getMaxRecursionLevel(request));
        if (filterXSS) {
            filterJSONObject(jsonObj, propertyName, xssApi);
        }
        response.getWriter().write(jsonObj.toString());
    }

    /**
     * Get recursion level from selectors. as per SLING-167: the last selector, if present, gives the recursion level.
     *
     * @param req the request
     * @return the recursion level
     * @throws IllegalArgumentException if the detected selector is not a number
     */
    protected static int getMaxRecursionLevel(SlingHttpServletRequest req) throws IllegalArgumentException {
        int maxRecursionLevels = 0;
        final String[] selectors = req.getRequestPathInfo().getSelectors();
        if (selectors.length > 0) {
            final String level = selectors[selectors.length - 1];
            if (INFINITY.equals(level)) {
                maxRecursionLevels = -1;
            } else {
                try {
                    maxRecursionLevels = Integer.parseInt(level);
                } catch (NumberFormatException nfe) {
                    if (StringUtils.isNumeric(level)) {
                        maxRecursionLevels = -1;
                    } else {
                        throw new IllegalArgumentException("Invalid recursion selector value '" + level + "'");
                    }
                }
            }
        }
        return maxRecursionLevels;
    }

    /**
     * Recursively filter properties in JSON Objects
     *
     * @param jsonObj      The JSON Object to filter
     * @param propertyName The name of the property that needs to be filtered
     * @param xssApi       A reference to the XSSAPI to use
     * @throws JSONException
     */
    protected static void filterJSONObject(JSONObject jsonObj, String propertyName, XSSAPI xssApi) throws JSONException {
        // only filter the 'propertyName' property if 'textIsRich' is true
        final boolean textIsRich = jsonObj.has(PROPERTY_TEXT_IS_RICH) && "true".equals(jsonObj.getString(PROPERTY_TEXT_IS_RICH));
        for (Iterator<String> keys = jsonObj.keys(); keys.hasNext();) {
            String k = keys.next();
            Object o = jsonObj.get(k);
            if ((o instanceof String) && k.equals(propertyName) && textIsRich) {
                // filter the property
                jsonObj.put(k, xssApi.filterHTML(jsonObj.getString(k)));
            } else if ((o instanceof JSONObject)) {
                // recursively filter all child objects
                filterJSONObject(jsonObj.getJSONObject(k), propertyName, xssApi);
            }
        }
    }
}