/*
 * Copyright 1997-2008 Day Management AG
 * Barfuesserplatz 6, 4001 Basel, Switzerland
 * All Rights Reserved.
 *
 * This software is the confidential and proprietary information of
 * Day Management AG, ("Confidential Information"). You shall not
 * disclose such Confidential Information and shall use it only in
 * accordance with the terms of the license agreement you entered into
 * with Day.
 */
package com.day.cq.wcm.msm.api;

import java.util.List;
import java.util.Set;

/**
 * Defines a Live Copy configuration between a blueprint and its target.
 * It can be seen as a configuration for a {@link  LiveRelationship}
 */
public interface LiveCopy {

    /**
     * Returns the path of the Live Copy
     * For example: "/content/copy"
     *
     * @return livecopy path
     */
    String getPath();

    /**
     * Returns the blueprint path of this Live Copy.
     * For example: "/content/geometrixx"
     *
     * @return the blueprint path
     */
    String getBlueprintPath();

    /**
     * Returns if the Live Copy is deep
     * @return <code>true</code> if deep
     */
    boolean isDeep();

    /**
     * Returns if the Live Copy is the root of a Live Copy tree
     * @return <code>true</code> if root
     */
    boolean isRoot();

    /**
     * Returns the list of {@link RolloutConfig} attached to the current <code>LiveCopy</code>
     * @return the list or an empty list if no {@link RolloutConfig}
     */
    List<RolloutConfig> getRolloutConfigs();

    /**
     * Returns the list of {@link RolloutConfig} attached to the current <code>LiveCopy</code> filtered by the
     * specified trigger.
     * @param trigger list is filtered with this trigger
     * @return the list or an empty list if no {@link RolloutConfig}
     */
    List<RolloutConfig> getRolloutConfigs(RolloutManager.Trigger trigger);

    /**
     * Returns if the current <code>LiveCopy</code> contains at least one {@link RolloutConfig} with the
     * specified trigger.
     * @param trigger the trigger to check
     * @return true if at least one config as specified trigger, false otherwise.
     */
    boolean isTrigger(RolloutManager.Trigger trigger);

    /**
     * A LiveCopy can be effective on its Resource or any ancestor of it, if set to be deep, as by {@link #isDeep() isDeep}.
     * This methods allows to omit ancestor Resources from the LiveCopy's Blueprint to be part of the LiveRelationship.<p>
     * There will be no immediate change to the Content of the LiveCopy, the change only become effective upon the
     * next roll-out
     * If the Resource already exists in the LiveCopy, it will no longer receive updates.
     * The {@link LiveRelationshipManager#hasLiveRelationship(org.apache.sling.api.resource.Resource)} will resolve to false,
     * even if the Resource once has been created in a Roll-out process
     * <p>
     * The exclusion is always effective on the Resource denoted by the path and all its descendants.
     *
     * @param relativePath paths relative to the {@link com.day.cq.wcm.msm.api.LiveCopy#getPath() LiveCopy's path }
     *                 that are should not be part of LiveRelationShip.
     * @return true in case the given Path has been {@link #contains(String) contained } before
     */
    boolean addExclusion(String relativePath);

    /**
     * Add back the Blueprint Resource at the given path to be part of the LiveRelationship defined by this LiveCopy.
     * There will be no immediate change to the Content of the LiveCopy, the change only becomes effective upon
     * the next roll-out
     * @param relativePath path relative to the {@link com.day.cq.wcm.msm.api.LiveCopy#getPath() LiveCopy's path } that
     *                 should take part in inheritance again
     * @return true in case the path has been {@link #contains(String) excluded } before
     * @see #addExclusion(String)
     */
    boolean removeExclusion(String relativePath);

    /**
     * Check if the given Path is covered by this LiveCopy.
     * Eg if LiveCopy is deep and the given path is descendant of this LiveCopy's path. And it is NOT part of the exclusion-list
     *
     * @param absPath the path to be considered, true if LiveCopy is deep and the path is not excluded
     *                or LiveCopy is not deep and path is equal to the LiveCopy's {@link #getPath() path}
     * @return true in case the given absolute path matches the LiveCopy's configuration.
     *         Generally speaking, is a descendant of the LiveCopy's path and not excluded.
     */
    boolean contains(String absPath);

    /**
     * All relative paths that have been set to be excluded from the LiveCopy by use of {@link #addExclusion(String)}.
     * This means for example that a shallow LiveCopy will return an empty Set though it does not
     * {@link #contains(String) include} any other path than the LiveCopy's.
     *
     * @return all relative paths that are explicitly excluded. The returned Set is NOT backed by the the LiveCopy nor
     *             will changes to the Set written back to the LiveCopy
     * @see #addExclusion(String)
     */
    Set<String> getExclusions();
}