/*
 * Copyright 1997-2008 Day Management AG
 * Barfuesserplatz 6, 4001 Basel, Switzerland
 * All Rights Reserved.
 *
 * This software is the confidential and proprietary information of
 * Day Management AG, ("Confidential Information"). You shall not
 * disclose such Confidential Information and shall use it only in
 * accordance with the terms of the license agreement you entered into
 * with Day.
 */
package com.day.cq.wcm.msm.api;

import java.util.Collection;
import java.util.Map;

import javax.jcr.RangeIterator;

import org.apache.sling.api.resource.Resource;
import org.apache.sling.api.resource.ResourceResolver;

import com.day.cq.wcm.api.Page;
import com.day.cq.wcm.api.WCMException;

public interface LiveRelationshipManager {

    /**
     * Checks if the given Resource is target of a LiveRelationship.
     * @param resource the resource to check
     * @return <code>true</code> if the Resource is controlled by a LiveCopy
     */
    boolean hasLiveRelationship(Resource resource);

    /**
     * Checks if the given source is the source of a Live Copy relationship.
     * @param resource the resource to check
     * @return <code>true</code> if the resource is a blueprint
     */
    boolean isSource(Resource resource);

    /**
     * Returns a Collection of LiveRelationships of the given LiveCopy source
     * Page to its targets. The search can be limited by trigger or target by
     * specifying the respective parameters. The given source page does not need
     * to be a LiveCopy root.
     *
     * @param source source page
     * @param trigger filter for triggers
     * @param targets filter for targets
     * @param advancedStatus True if advanced status (time consuming) must be computed for relationships
     * @return the relationships
     * @throws WCMException if an error occurs
     * @deprecated since 5.7, use {@link #getLiveRelationships(org.apache.sling.api.resource.Resource, String, com.day.cq.wcm.msm.api.RolloutManager.Trigger)}
     */
    Collection<LiveRelationship> getLiveRelationships(Page source,
                                                RolloutManager.Trigger trigger,
                                                String[] targets,
                                                boolean advancedStatus) throws WCMException;

    /**
     * Returns a collection of LiveRelationships of the given LiveCopy's source
     * to its targets. The search can be limited by trigger or target by
     * specifying the respective parameters. The given source resource does
     * not need to be a {@link LiveCopy#isRoot() LiveCopy root} .
     *
     * @param source resource
     * @param trigger filter for triggers
     * @param targets filter for targets
     * @param advancedStatus True if advanced status (time consuming) must be computed for relationships
     * @return the relationships
     * @throws WCMException if an error occurs
     * @deprecated since 5.7, use {@link #getLiveRelationships(org.apache.sling.api.resource.Resource, String, com.day.cq.wcm.msm.api.RolloutManager.Trigger)}
     */
    Collection<LiveRelationship> getLiveRelationships(Resource source,
                                                RolloutManager.Trigger trigger,
                                                String[] targets,
                                                boolean advancedStatus) throws WCMException;

    /**
     * Find all LiveRelationships, that have the given Resource as Blueprint.
     * The result can be filtered to match a branch with giving a target filter and further an optional filter for the
     * Trigger.<br>
     * <br>
     * The result is a RangeIterator that contains the matching LiveRelationships.
     *
     * @param source to inspect
     * @param targetPathFilter absolute path the LiveRelationships should be searched in, null indicates root
     *                         The path filter matches always self and descendants
     * @param triggerFilter the Trigger the LiveRelationships must have
     *                      {@link LiveRelationship#isTrigger(com.day.cq.wcm.msm.api.RolloutManager.Trigger)}.
     *                      null argument indicates any
     * @return hits
     * @throws WCMException in case of an error
     * @see RangeIterator
     */
    RangeIterator getLiveRelationships(Resource source, String targetPathFilter, RolloutManager.Trigger triggerFilter)
            throws WCMException;


    /**
     * Returns a {@link com.day.cq.wcm.msm.api.LiveRelationship LiveRelationship}
     * of the given {@link com.day.cq.wcm.msm.api.LiveCopy LiveCopy} Resource
     *
     * @param target the LiveCopy target Resource
     * @param advancedStatus True if advanced status must be computed for relationship
     * @return the relationship or <code>null</code> if the resource is not a LiveCopy
     * @throws WCMException if an error occurs
     */
    LiveRelationship getLiveRelationship(Resource target, boolean advancedStatus) throws WCMException;

    /**
     * Get the children of the given LiveRelationship<br>
     * <br>
     * This is the combined set  of the Blueprint's and the LiveRelationship's children Resources.
     * Thus the result contains the Relationships for Resources that only exists in Blueprint, that only exists in LiveCopy
     * and that exists in both.<br>
     * <br>
     * The order of result is not specific.
     *
     * @param relationship to access the children for
     * @param resourceResolver to use to access the resources with
     * @return the combined set of LiveRelationships
     * @throws WCMException in case of an error
     * @see RangeIterator
     */
    RangeIterator getChildren(LiveRelationship relationship, ResourceResolver resourceResolver) throws WCMException;


    /**
     * Puts a Page under the control of a LiveCopy
     *
     * @param source the source to be synced to the copy
     * @param copy the page to receive the changes from the source
     * @param deep if false the LiveCopy of this relation ship will only be valid for the Page given as copy argument
     * @param autoSave if the changes should be committed
     * @param rolloutConfig the configuration to be use for the synchronization
     * @return the established LiveRelationship
     * @throws WCMException in case of error establishing the relationship
     */
    LiveRelationship establishRelationship(Page source, Page copy, boolean deep, boolean autoSave, RolloutConfig...rolloutConfig) throws WCMException;

    /**
     * Ends the {@link com.day.cq.wcm.msm.api.LiveRelationship LiveRelationship}
     * of the given Resource.<br>
     * The Resource has to be part of a {@link com.day.cq.wcm.msm.api.LiveCopy LiveCopy}
     * As indicated by {@link #hasLiveRelationship(org.apache.sling.api.resource.Resource) hasLiveRelationship method}
     * <br>
     * A detached Resource is treated as any other Resource that is created in
     * the LiveCopy tree.<br>
     * NOTE: this means especially that any history of this relation is gone.
     * If you want to maintain this history or just want to keep the current
     * version of the Resource use the {@link #cancelRelationship(org.apache.sling.api.resource.ResourceResolver, LiveRelationship, boolean, boolean) cancelRelationship method}
     *
     * @param resource to detach from LiveRelationship
     * @param autoSave if true changes are committed immediately
     * @throws WCMException in case an error occurs on detachment
     * @see #hasLiveRelationship(Resource)
     */
    void endRelationship(Resource resource, boolean autoSave) throws WCMException;

    /**
     * Cancel the relationship of the given relation.
     * @param resolver resource resolver
     * @param relation relation
     * @param deep if <code>true</code> all child relations are canceled, too
     * @param autoSave  Save modifications
     * @throws WCMException if an error occurs
     */
    void cancelRelationship(ResourceResolver resolver,
                            LiveRelationship relation,
                            boolean deep,
                            boolean autoSave) throws WCMException;

    /**
     * Re-enable the relationship of a canceled one.
     * @param resolver resource resolver
     * @param relation relation to re-enable
     * @param autoSave  Save modifications
     * @throws WCMException if an error occurs
     */
    void reenableRelationship(ResourceResolver resolver, LiveRelationship relation, boolean autoSave)
            throws WCMException;

    /**
     * Cancel the relationship of the given relation.
     * @param resolver resource resolver
     * @param relation relation
     * @param names list of property names for which inheritance is canceled
     * @param autoSave  Save modifications
     * @throws WCMException if an error occurs
     */
    void cancelPropertyRelationship(ResourceResolver resolver,
            LiveRelationship relation,
            String[] names,
            boolean autoSave) throws WCMException;

    /**
     * Re-enable the relationship of a canceled one.
     * @param resolver resource resolver
     * @param relation relation to re-enable
     * @param names list of property names for which inheritance is reenabled
     * @param autoSave  Save modifications
     * @throws WCMException if an error occurs
     */
    void reenablePropertyRelationship(ResourceResolver resolver,
            LiveRelationship relation, String[] names, boolean autoSave) throws WCMException;

    /**
     * Returns a map of pairs: Live Copy skipped page's path / <b>Blueprint page</b>. Each Live Copy skipped page's
     * path is the path of a skipped child of the provided page and returned map contains its
     * corresponding Blueprint page.
     *
     * @param parentTargetPage Parent target page
     * @return the map
     * @throws WCMException if an error occurs
     * @deprecated  since 5.7 use {@link LiveCopy#getExclusions()} instead
     */
    @Deprecated
    public Map<String, Page> getSkippedSourcePages(Page parentTargetPage) throws WCMException;

    /**
     * Adds several pages to the skipped pages list (stored on parent page).
     *
     * @param parentPage Parent page
     * @param childName names of the pages to be skipped
     * @param autoSave   Save modifications
     * @throws WCMException if an error occurs
     * @see Page#getName()
     * @deprecated  since 5.7 use {@link LiveCopy#addExclusion(String)} instead
     */
    @Deprecated
    public void addSkippedPages(Page parentPage, String[] childName, boolean autoSave) throws WCMException;

    /**
     * Removes several pages from the skipped pages list (stored on parent page).
     *
     * @param parentPage Parent page
     * @param childName names of the Pages to be unskipped
     * @param autoSave   Save modifications
     * @throws WCMException if an error occurs
     * @see Page#getName()
     * @deprecated  since 5.7 use {@link LiveCopy#removeExclusion(String)} instead
     */
    @Deprecated
    public void removeSkippedPages(Page parentPage, String[] childName, boolean autoSave) throws WCMException;

    /**
     * Returns all live copies. the Live Copy path is used as key of the map.
     * @return all live copies.
     * @throws WCMException in case of error accessing the LiveCopy
     * @deprecated since 5.5 use {@link #getLiveRelationships(org.apache.sling.api.resource.Resource, com.day.cq.wcm.msm.api.RolloutManager.Trigger, String[], boolean)}
     *             to search for LiveCopies
     */
    @Deprecated
    Map<String, LiveCopy> getLiveCopies() throws WCMException;

    /**
     * Returns the closest Live Copy corresponding the <code>target</code> resource.
     * @param target resource
     * @return Live Copy or null if no Live Copy found.
     * @throws WCMException in case of error accessing the LiveCopy
     * @deprecated since 5.5 user {@link #getLiveRelationship(org.apache.sling.api.resource.Resource, boolean)}
     */
    @Deprecated
    LiveCopy getLiveCopy(Resource target) throws WCMException;

    /**
     * Checks if the given source is in a blueprint tree.
     * @param source the resource to check
     * @return <code>true</code> if the resource is a blueprint
     * @deprecated use {@link BlueprintManager#getContainingBlueprint(String)} to check
     */
    @Deprecated
    boolean isInBlueprint(Resource source);

    /**
     * Checks if the given sync target is part of a Live Copy relationship.
     * @param target the resource to check
     * @return <code>true</code> if the resource is live synced
     * @deprecated since 5.5 use {@link #hasLiveRelationship(org.apache.sling.api.resource.Resource)} instead
     */
    @Deprecated
    boolean isLiveCopy(Resource target);

    /**
     * Ends the {@link com.day.cq.wcm.msm.api.LiveRelationship LiveRelationship}
     * of the given Resource.<br>
     * The Resource has to be part of a {@link com.day.cq.wcm.msm.api.LiveCopy LiveCopy}
     * As indicated by {@link #hasLiveRelationship(org.apache.sling.api.resource.Resource) hasLiveRelationship method}
     * <br>
     * A detached Resource is treated as any other Resource that is created in
     * the LiveCopy tree.<br>
     * NOTE: this means especially that any history of this relation is gone.
     * If you want to maintain this history or just want to keep the current
     * version of the Resource use the {@link #cancelRelationship(org.apache.sling.api.resource.ResourceResolver, LiveRelationship, boolean, boolean) method}
     *
     * @param resource to detach from LiveRelationship
     * @param autoSave if true changes are committed immediately
     * @throws WCMException in case an error occurs on detachment
     * @see #isLiveCopy(Resource)
     * @deprecated since 5.5 use endRelationship()
     */
    @Deprecated
    void detach(Resource resource, boolean autoSave) throws WCMException;
}