/*
 * Copyright 1997-2008 Day Management AG
 * Barfuesserplatz 6, 4001 Basel, Switzerland
 * All Rights Reserved.
 *
 * This software is the confidential and proprietary information of
 * Day Management AG, ("Confidential Information"). You shall not
 * disclose such Confidential Information and shall use it only in
 * accordance with the terms of the license agreement you entered into
 * with Day.
 */
package com.day.cq.workflow;

import java.security.AccessControlException;
import java.util.Dictionary;
import java.util.List;
import java.util.Map;

import javax.jcr.Session;
import javax.jcr.version.VersionException;

import org.apache.jackrabbit.api.security.user.Authorizable;

import com.day.cq.workflow.collection.util.ResultSet;
import com.day.cq.workflow.exec.HistoryItem;
import com.day.cq.workflow.exec.Route;
import com.day.cq.workflow.exec.WorkItem;
import com.day.cq.workflow.exec.Workflow;
import com.day.cq.workflow.exec.WorkflowData;
import com.day.cq.workflow.exec.filter.WorkItemFilter;
import com.day.cq.workflow.model.WorkflowModel;
import com.day.cq.workflow.model.WorkflowModelFilter;
import com.day.cq.workflow.model.WorkflowNode;

/**
 * The <code>WorkflowSession</code> class provides all functionality (depending
 * on the users rights) for managing {@link WorkflowModel}s, {@link Workflow}
 * instances and their execution.
 * 
 * <p>
 * It provides methods to:
 * 
 * <ul>
 * <li>deploy/create a workflow models</li>
 * <li>start, stop, suspend, resume and complete a workflow instance</li>
 * <li>retrieve and complete {@link WorkItem}s assigned to a user/group</li>
 * </ul>
 * 
 */
public interface WorkflowSession {
    /**
     * Returns the <code>{@link WorkflowService}</code> that was used to create
     * the <code>WorkflowSession</code> instance.
     * 
     * @return The <code>{@link WorkflowService}</code> that was used to create
     *         the <code>WorkflowSession</code> instance.
     */
    WorkflowService getWorkflowService();

    /**
     * Returns the JCR <code>{@link Session}</code> assigned to the
     * <code>WorkflowSession</code> instance.
     * 
     * @return The JCR <code>{@link Session}</code> assigned to the
     *         <code>WorkflowSession</code> instance.
     */
    Session getSession();

    /**
     * Returns the user that owns the <code>WorkflowSession</code>.
     * 
     * @return The user that owns the <code>WorkflowSession</code>.
     */
    Authorizable getUser();

    /**
     * Evaluate the passed rule by choosing the right rule engine.
     * 
     * @param data
     *            workflow data to apply the rule
     * @param rule
     *            rule to execute
     * @return <code>true</code> if rule evaluation is true or the rule is
     *         <code>null</code>
     */
    boolean evaluate(WorkflowData data, String rule);

    /**
     * Deploys a new or modified <code>{@link WorkflowModel}</code>.
     * 
     * @param model
     *            The <code>{@link WorkflowModel}</code> to be deployed.
     * @throws WorkflowException
     *             Thrown in case an error prevents deployment of the
     *             <code>{@link WorkflowModel}</code>.
     */
    void deployModel(WorkflowModel model) throws WorkflowException;

    /**
     * Creates a new blank <code>{@link WorkflowModel}</code>.
     * 
     * @param title
     *            The title of the newly created
     *            <code>{@link WorkflowModel}</code>.
     * 
     * @return The newly created <code>{@link WorkflowModel}</code>.
     * 
     * @throws WorkflowException
     *             Thrown in case an error prevents creation of the
     *             <code>{@link WorkflowModel}</code>.
     */
    WorkflowModel createNewModel(String title) throws WorkflowException;
    
    /**
     * Creates a new blank <code>{@link WorkflowModel}</code> under the given path.
     * 
     * @param title
     *            The title of the newly created
     *            <code>{@link WorkflowModel}</code>.
     * @param id
     *            The ID of the new <code>{@link WorkflowModel}</code>.
     * 
     * @return The newly created <code>{@link WorkflowModel}</code>.
     * 
     * @throws WorkflowException
     *             Thrown in case an error prevents creation of the
     *             <code>{@link WorkflowModel}</code>.
     */
    WorkflowModel createNewModel(String title, String id) throws WorkflowException;

    /**
     * Deletes the given <code>{@link WorkflowModel}</code>.
     * 
     * @param id
     *            The ID of the <code>{@link WorkflowModel}</code> to be
     *            deleted.
     * @throws WorkflowException
     *             Thrown in case an error prevents deployment of the
     *             <code>{@link WorkflowModel}</code>.
     */
    void deleteModel(String id) throws WorkflowException;

    /**
     * Returns newest versions of all deployed
     * <code>{@link WorkflowModel}</code>s.
     * 
     * @return All deployed <code>{@link WorkflowModel}</code> at their newest
     *         versions.
     * 
     * @throws WorkflowException
     *             Thrown in case an error occurs while fetching the
     *             <code>{@link WorkflowModel}</code>s.
     */
    WorkflowModel[] getModels() throws WorkflowException;
    
    /**
     * Returns newest versions of all deployed
     * <code>{@link WorkflowModel}</code>s.
     *
     * @param filter The <code>{@link WorkflowModelFilter}</code> used for getting workflow models
     *
     * @return All deployed <code>{@link WorkflowModel}</code> at their newest
     *         versions.
     * 
     * @throws WorkflowException
     *             Thrown in case an error occurs while fetching the
     *             <code>{@link WorkflowModel}</code>.
     */
    WorkflowModel[] getModels(WorkflowModelFilter filter) throws WorkflowException;
    
    /**
     * Returns newest versions of all deployed
     * <code>{@link WorkflowModel}</code>s.
     * 
     * 
     * @param start
     *            Start index of the result set.
     * @param limit
     *            Limit of the result set.
     * 
     * 
     * @return All deployed <code>{@link WorkflowModel}</code> at their newest
     *         versions.
     * 
     * @throws WorkflowException
     *             Thrown in case an error occurs while fetching the
     *             <code>{@link WorkflowModel}</code>s.
     */
    ResultSet<WorkflowModel> getModels(long start, long limit) throws WorkflowException;
    
    /**
     * Returns newest versions of all deployed
     * <code>{@link WorkflowModel}</code>s.
     * 
     * 
     * @param start
     *            Start index of the result set.
     * @param limit
     *            Limit of the result set.
     *
     * @param filter The <code>{@link WorkflowModelFilter}</code> used for getting workflow models
     *
     * @return All deployed <code>{@link WorkflowModel}</code> at their newest
     *         versions.
     * 
     * @throws WorkflowException
     *             Thrown in case an error occurs while fetching the
     *             <code>{@link WorkflowModel}</code>s.
     */
    ResultSet<WorkflowModel> getModels(long start, long limit, WorkflowModelFilter filter) throws WorkflowException;

    /**
     * Returns newest version of the <code>{@link WorkflowModel}</code> with the
     * given ID.
     * 
     * @param id
     *            The ID of the {@link WorkflowModel}.
     * 
     * @return The newest version of the <code>{@link WorkflowModel}</code> with
     *         the given ID or null, if no <code>{@link WorkflowModel}</code>
     *         with the given ID exists.
     * 
     * @throws WorkflowException
     *             Thrown in case an error occurs while fetching the
     *             <code>{@link WorkflowModel}</code>.
     */
    WorkflowModel getModel(String id) throws WorkflowException;

    /**
     * Returns the <code>{@link WorkflowModel}</code> with the given ID and
     * version.
     * 
     * @param id
     *            The ID of the {@link WorkflowModel}.
     * @param version
     *            The version of the {@link WorkflowModel}.
     * 
     * @return The <code>{@link WorkflowModel}</code> with the given ID and
     *         version or null, if no <code>{@link WorkflowModel}</code> matches
     *         the given ID and/or version.
     * 
     * @throws WorkflowException
     *             Thrown in case an error occurs while fetching the
     *             <code>{@link WorkflowModel}</code>.
     * @throws VersionException
     *             Thrown in case the version does not exist or can not be
     *             accessed.
     */
    WorkflowModel getModel(String id, String version) throws WorkflowException,
            VersionException;

    /**
     * Start a new {@link Workflow} instance with the given
     * <code>{@link WorkflowModel}</code> and <code>{@link WorkflowData}</code>.
     * 
     * @param model
     *            <code>{@link WorkflowModel}</code> to be used for the new
     *            {@link Workflow} instance.
     * @param data
     *            <code>{@link WorkflowData}</code> to be attached to the new
     *            workflow instance.
     * @return The <code>{@link Workflow}</code> of the newly started
     *         {@link Workflow} instance.
     * 
     * @throws WorkflowException
     *             Thrown in case the workflow process cannot be started.
     */
    Workflow startWorkflow(WorkflowModel model, WorkflowData data)
            throws WorkflowException;

    /**
     * Start a new {@link Workflow} instance with the given
     * <code>{@link WorkflowModel}</code> and <code>{@link WorkflowData}</code>.
     * 
     * @param model
     *            <code>{@link WorkflowModel}</code> to be used for the new
     *            {@link Workflow} instance.
     * @param data
     *            <code>{@link WorkflowData}</code> to be attached to the new
     *            workflow instance.
     * @param metaData
     *            Meta data assigned to the new <code>{@link Workflow}</code>.
     * @return The <code>{@link Workflow}</code> of the newly started
     *         {@link Workflow} instance.
     * 
     * @throws WorkflowException
     *             Thrown in case the workflow process cannot be started.
     *             
     * @deprecated It is recommended to use startWorkflow(WorkflowModel, WorkflowData, Map) instead.
     */
    Workflow startWorkflow(WorkflowModel model, WorkflowData data,
            Dictionary<String, String> metaData) throws WorkflowException;
    
    /**
     * Start a new {@link Workflow} instance with the given
     * <code>{@link WorkflowModel}</code>, <code>{@link WorkflowData}</code> and <code>meta data</code>.
     * 
     * @param model
     *            <code>{@link WorkflowModel}</code> to be used for the new
     *            {@link Workflow} instance.
     * @param data
     *            <code>{@link WorkflowData}</code> to be attached to the new
     *            workflow instance.
     * @param metaData
     *            Meta data assigned to the new <code>{@link Workflow}</code>.
     * @return The <code>{@link Workflow}</code> of the newly started
     *         {@link Workflow} instance.
     * 
     * @throws WorkflowException
     *             Thrown in case the workflow process cannot be started.
     */
    Workflow startWorkflow(WorkflowModel model, WorkflowData data,
            Map<String, Object> metaData) throws WorkflowException;

    /**
     * Terminates the given {@link Workflow} instance.
     * 
     * @param instance
     *            The <code>{@link Workflow}</code> instance that shall be
     *            terminated.
     * 
     * @throws WorkflowException
     *             Thrown in case the workflow instance cannot be terminated.
     */
    void terminateWorkflow(Workflow instance) throws WorkflowException;

    /**
     * Resumes the given {@link Workflow} instance.
     * 
     * @param instance
     *            The <code>{@link Workflow}</code> instance that shall be
     *            resumed.
     * 
     * @throws WorkflowException
     *             Thrown in case the workflow instance cannot be resumed.
     */
    void resumeWorkflow(Workflow instance) throws WorkflowException;

    /**
     * Suspends the given {@link Workflow} instance.
     * 
     * @param instance
     *            The <code>{@link Workflow}</code> of the workflow instance
     *            that shall be suspended.
     * 
     * @throws WorkflowException
     *             Thrown in case the workflow instance cannot be suspended.
     */
    void suspendWorkflow(Workflow instance) throws WorkflowException;

    /**
     * All <code>{@link WorkItem}</code>s assigned to an active {@link Workflow}
     * instance and the respective user of the <code>WorkflowSession</code> will
     * be returned.
     * 
     * @return All <code>{@link WorkItem}</code>s assigned to an active workflow
     *         and the respective users <code>WorkflowSession</code>.
     * 
     * @throws WorkflowException
     *             Thrown in case an exception occurs while fetching the active
     *             <code>{@link WorkItem}</code>s.
     */
    WorkItem[] getActiveWorkItems() throws WorkflowException;

    /**
     * All <code>{@link WorkItem}</code>s assigned to an active {@link Workflow}
     * instance and the respective user of the <code>WorkflowSession</code> will
     * be returned.
     * 
     * @param start
     *            Start index of the result set.
     * @param limit
     *            Limit of the result set.
     * 
     * @return All <code>{@link WorkItem}</code>s assigned to an active workflow
     *         and the respective users <code>WorkflowSession</code>.
     * 
     * @throws WorkflowException
     *             Thrown in case an exception occurs while fetching the active
     *             <code>{@link WorkItem}</code>s.
     */
    ResultSet<WorkItem> getActiveWorkItems(long start, long limit)
            throws WorkflowException;
    
    /**
     * All <code>{@link WorkItem}</code>s assigned to an active {@link Workflow}
     * instance and the respective user of the <code>WorkflowSession</code> will
     * be returned.
     * 
     * @param start
     *            Start index of the result set.
     * @param limit
     *            Limit of the result set.
     * @param filter
     *            filter for work items
     * 
     * @return All <code>{@link WorkItem}</code>s assigned to an active workflow
     *         and the respective users <code>WorkflowSession</code>.
     * 
     * @throws WorkflowException
     *             Thrown in case an exception occurs while fetching the active
     *             <code>{@link WorkItem}</code>s.
     */
    ResultSet<WorkItem> getActiveWorkItems(long start, long limit, WorkItemFilter filter)
            throws WorkflowException;
    
    /**
     * All <code>{@link WorkItem}</code>s assigned to a {@link Workflow}
     * instance and assigned to the respective user of the
     * <code>WorkflowSession</code> will be returned.
     * 
     * @return All <code>{@link WorkItem}</code>s assigned to a {@link Workflow}
     *         instance and the respective users <code>WorkflowSession</code>.
     * 
     * @throws WorkflowException
     *             Thrown in case an exception occurs while fetching all
     *             <code>{@link WorkItem}</code>s.
     */
    WorkItem[] getAllWorkItems() throws WorkflowException;

    /**
     * All <code>{@link WorkItem}</code>s assigned to a {@link Workflow}
     * instance and assigned to the respective user of the
     * <code>WorkflowSession</code> will be returned.
     * 
     * @param start
     *            Start index of the result set.
     * @param limit
     *            Limit of the result set.
     * 
     * @return All <code>{@link WorkItem}</code>s assigned to a {@link Workflow}
     *         instance and the respective users <code>WorkflowSession</code>.
     * 
     * @throws WorkflowException
     *             Thrown in case an exception occurs while fetching all
     *             <code>{@link WorkItem}</code>s.
     */
    ResultSet<WorkItem> getAllWorkItems(long start, long limit)
            throws WorkflowException;

    /**
     * Returns the <code>{@link WorkItem}</code> with the given ID assigned to a
     * {@link Workflow} instance, but only if the respective users
     * <code>WorkflowSession}</code> has access to it.
     * 
     * @param id
     *            The ID of the <code>{@link WorkItem}</code>.
     * @return The <code>{@link WorkItem}</code> with the given ID.
     * @throws WorkflowException
     *             Thrown in case an exception is occurs while fetching the
     *             <code>{@link WorkItem}</code>.
     */
    WorkItem getWorkItem(String id) throws WorkflowException;

    /**
     * Returns all <code>{@link Workflow}</code> instances that are in one of
	 * the given states and to which the
	 * <code>{@link WorkflowSession}</code> has access to.
     * 
     * @param states
     *            The list of states used to select the
     *            <code>{@link Workflow}</code> instances.
     * 
     * @return All <code>{@link Workflow}</code>s that are in one of the given
     *         states and to which the <code>{@link WorkflowSession}</code> has
     *         access to.
     * 
     * @throws WorkflowException
     *             Thrown in case an error occurred while retrieving the
     *             <code>{@link Workflow}</code>s.
     */
    Workflow[] getWorkflows(String[] states) throws WorkflowException;

    /**
     * Returns all <code>{@link Workflow}</code>s instances that are in one of
     * the given states and to which the
     * <code>{@link WorkflowSession}</code> has access to.
     * 
     * @param states
     *            The list of states used to select the
     *            <code>{@link Workflow}</code> instances.
     * @param start
     *            Start index of the result set.
     * @param limit
     *            Limit of the result set.
     * 
     * @return All <code>{@link Workflow}</code>s that are in one of the given
     *         states and to which the <code>{@link WorkflowSession}</code> has
     *         access to.
     * 
     * @throws WorkflowException
     *             Thrown in case an error occurred while retrieving the
     *             <code>{@link Workflow}</code>s.
     */
    ResultSet<Workflow> getWorkflows(String[] states, long start, long limit)
            throws WorkflowException;

    /**
     * Returns all {@link Workflow}s which the
     * <code>{@link WorkflowSession}</code> has access to.
     * 
     * @return All <code>{@link Workflow}</code>s the
     *         <code>{@link WorkflowSession}</code> has access to.
     * 
     * @throws WorkflowException
     *             Thrown in case an error occurred while retrieving the
     *             <code>{@link Workflow}</code>s.
     */
    Workflow[] getAllWorkflows() throws WorkflowException;

    /**
     * Retrieve the <code>{@link Workflow}</code> instance with the given ID.
     * 
     * @param id
     *            The ID of the <code>{@link Workflow}</code> instance that
     *            should be returned.
     * 
     * @return The <code>{@link Workflow}</code> instance with the given ID or
     *         null, if no <code>{@link Workflow}</code> instance ID matches the
     *         given ID.
     * 
     * @throws WorkflowException
     *             Thrown in case an unexpected error occurs while retrieving
     *             the <code>{@link Workflow}</code>.
     */
    Workflow getWorkflow(String id) throws WorkflowException;

    /**
     * Completes the <code>{@link WorkItem}</code> for the assigned
     * <code>{@link Workflow}</code> instance and assign to the <i>next</i>
     * <code>{@link WorkflowNode}</code> according to the given
     * <code>{@link Route}</code>.
     * 
     * @param item
     *            The <code>{@link WorkItem}</code> to be completed.
     * @param route
     *            The <code>{@link Route}</code> to the destination to be
     *            advanced.
     * 
     * @throws WorkflowException
     *             Thrown in case an error is occurs while completing the
     *             <code>{@link WorkItem}</code>.
     */
    void complete(WorkItem item, Route route) throws WorkflowException;

    /**
     * All <code>{@link Route}</code> to all possible destinations will be
     * returned. Rules will be already executed, thus all routes that are active
     * can be selected.
     * 
     * @param item
     *            The <code>{@link WorkItem}</code> to retrieve routes from.
     * 
     * @return All <code>{@link Route}</code>s to advance.
     * 
     * @throws WorkflowException
     *             Thrown in case an error is occurs while evaluating the
     *             <code>{@link Route}</code>s.
     */
    List<Route> getRoutes(WorkItem item) throws WorkflowException;

    /**
     * All <code>{@link Route}</code> to all possible destinations will be
     * returned. Rules will be already executed, thus all routes that are active
     * can be selected.
     * 
     * @param item
     *            The <code>{@link WorkItem}</code> to retrieve routes from.
     * 
     * @param expand
     *            Expands the group members if the destination points to a
     *            group. Means that not only the group is retourned as route but
     *            as well all group members.
     * 
     * @return All <code>{@link Route}</code>s to advance.
     * 
     * @throws WorkflowException
     *             Thrown in case an error is occurs while evaluating the
     *             <code>{@link Route}</code>s.
     */
    List<Route> getRoutes(WorkItem item, boolean expand)
            throws WorkflowException;

    /**
     * All <code>{@link Route}</code> to all possible already passed
     * destinations will be returned. Rules will be already executed, thus all
     * routes that are active can be selected.
     * 
     * @param item
     *            The <code>{@link WorkItem}</code> to retrieve back routes
     *            from.
     * 
     * @return All <code>{@link Route}</code>s to advance (back).
     * 
     * @throws WorkflowException
     *             Thrown in case an error is occurs while evaluating the
     *             <code>{@link Route}</code>s.
     */
    List<Route> getBackRoutes(WorkItem item) throws WorkflowException;

    /**
     * All <code>{@link Route}</code> to all possible already passed
     * destinations will be returned. Rules will be already executed, thus all
     * routes that are active can be selected.
     * 
     * @param item
     *            The <code>{@link WorkItem}</code> to retrieve back routes
     *            from.
     * 
     * @param expand
     *            Expands the group members if the destination points to a
     *            group. Means that not only the group is retourned as route but
     *            as well all group members.
     * 
     * @return All <code>{@link Route}</code>s to advance (back).
     * 
     * @throws WorkflowException
     *             Thrown in case an error is occurs while evaluating the
     *             <code>{@link Route}</code>s.
     */
    List<Route> getBackRoutes(WorkItem item, boolean expand)
            throws WorkflowException;

    /**
     * Creates a new <code>{@link WorkflowData}</code> instance based on the
     * given data.
     * 
     * @param payloadType
     *            The type of payload for the new
     *            <code>{@link WorkflowData}</code> instance.
     * @param payload
     *            The payload object used for creating the new
     *            <code>{@link WorkflowData}</code> instance.
     *            <code>{@link WorkflowData}</code> instance.
     * 
     * @return The new <code>{@link WorkflowData}</code> instance.
     */
    WorkflowData newWorkflowData(String payloadType, Object payload);

    /**
     * Returns all <code>delegatees</code> as <code>Authorizable</code> who are
     * allowed to take over the <code>WorkItem</code>
     * 
     * @param item
     *            The workitem
     * 
     * @return The list of <code>Authorizable</code>s who are allowed to act as
     *         delegatees.
     * 
     * @throws WorkflowException
     *             in case the delegatees could not be retrieved for some reason
     */
    List<Authorizable> getDelegatees(WorkItem item) throws WorkflowException;

    /**
     * Delegates the <code>WorkItem</code> to the specified
     * <code>Authorizable</code>.
     * 
     * @param item
     *            the corresponding workitem
     * @param participant
     *            the participant to delegate
     * @throws java.security.AccessControlException
     *             if the given participant is not allowed to take over the
     *             workitem.
     * @throws WorkflowException
     *             in case the delegation fails in regards of persisting the
     *             "new" state.
     */
    void delegateWorkItem(WorkItem item, Authorizable participant)
            throws WorkflowException, AccessControlException;

    /**
     * The complete ordered list of {@link com.day.cq.workflow.exec.HistoryItem}
     * s is returned for the given {@link Workflow} instance.
     * 
     * @param instance
     *            {@link Workflow} instance
     * 
     * @return <code>List</code> of {@link com.day.cq.workflow.exec.HistoryItem}
     *         s
     * 
     * @throws WorkflowException
     *             in case the history item retrieval fails
     */
    List<HistoryItem> getHistory(Workflow instance) throws WorkflowException;

    /**
     * Updates the workflow data of a running workflow instance to the given
     * <code>WorkflowData</code>.
     * 
     * @param instance
     *            The <code>{@link Workflow}</code> instance for which to update the data.
     * @param data
     *            The <code>{@link WorkflowData}</code> for the update.
     */
    void updateWorkflowData(Workflow instance, WorkflowData data);

    /**
     * The logout method has to be called in order to clean up all jcr session
     * references.
     */
    void logout();

    /**
     * Checks for the workflow superuser
     * 
     * @return <code>true</code> if the user is a workflow superuser
     */
    boolean isSuperuser();

    /**
     * This method restarts a workflow instance that is in a "stale" state.
     * means there is no workitem around. this could happen in case the eventing
     * which controls the the script/java processes purges an event/job (there
     * is the depending (serialized) workitem stored). The restart tries to
     * start the workflow at the position right before the workitem got lost.
     * <b>Note</b>: This is an administrative method and can only be used by a
     * workflow superuser!
     * 
     * @param workflow
     *            workflow
     * @throws WorkflowException
     *             in case an error occurs
     */
    void restartWorkflow(Workflow workflow) throws WorkflowException;
}
