/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * ___________________
 *
 *  Copyright 2012 Adobe Systems Incorporated
 *  All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Systems Incorporated and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Systems Incorporated.
 **************************************************************************/
package com.day.cq.workflow.compatibility;

import com.adobe.granite.workflow.WorkflowException;
import com.adobe.granite.workflow.WorkflowSession;
import com.adobe.granite.workflow.exec.StepExecutor;
import com.adobe.granite.workflow.exec.WorkItem;
import com.adobe.granite.workflow.metadata.MetaDataMap;
import com.day.cq.workflow.WorkflowService;
import com.day.cq.workflow.exec.JavaProcess;
import com.day.cq.workflow.exec.JavaProcessExt;
import com.day.cq.workflow.exec.WorkflowProcess;
import com.day.cq.workflow.impl.CQWorkflowSessionWrapper;
import com.day.cq.workflow.impl.exec.CQWorkItemWrapper;
import com.day.cq.workflow.impl.metadata.CQMetaDataMap;
import com.day.cq.workflow.impl.model.CQWorkflowNodeWrapper;
import org.apache.felix.scr.annotations.Component;
import org.apache.felix.scr.annotations.Property;
import org.apache.felix.scr.annotations.Reference;
import org.apache.felix.scr.annotations.ReferenceCardinality;
import org.apache.felix.scr.annotations.ReferencePolicy;
import org.apache.felix.scr.annotations.References;
import org.apache.felix.scr.annotations.Service;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.List;
import java.util.concurrent.CopyOnWriteArrayList;

@Component(metatype = false, label = "%cq.workflow.compat.executor.name", description = "%cq.workflow.compat.executor.description")
@Property(name = "service.description", value="%cq.workflow.compat.executor.description")
@References({
    @Reference(name = "WorkflowProcess", cardinality = ReferenceCardinality.OPTIONAL_MULTIPLE, referenceInterface = WorkflowProcess.class, policy = ReferencePolicy.DYNAMIC),
    @Reference(name = "Process", cardinality = ReferenceCardinality.OPTIONAL_MULTIPLE, referenceInterface = JavaProcess.class, policy = ReferencePolicy.DYNAMIC)
})
@Service
public class CQWorkflowProcessRunner implements StepExecutor {
    protected final Logger log = LoggerFactory.getLogger(getClass());

    @Reference
    private WorkflowService cqWorkflowService;

    protected List<WorkflowProcess> workflowProcesses = new CopyOnWriteArrayList<WorkflowProcess>();
    protected List<JavaProcess> processes = new CopyOnWriteArrayList<JavaProcess>();

    public void bindWorkflowProcess(WorkflowProcess process) {
        workflowProcesses.add(process);
    }

    public void unbindWorkflowProcess(WorkflowProcess process) {
        workflowProcesses.remove(process);
    }

    // TODO remove after 5.4
    public void bindProcess(JavaProcess process) {
        processes.add(process);
    }

    // TODO remove after 5.4
    public void unbindProcess(JavaProcess process) {
        processes.remove(process);
    }

    public void execute(String resource, WorkItem workItem, WorkflowSession workflowSession, MetaDataMap metaDataMap) throws WorkflowException {
        // search for process implementation to be executed
        Object toBeExecuted = evaluateExecutable(resource);
        if (toBeExecuted != null) {
            try {
                CQWorkflowSessionWrapper sessionWrapper = new CQWorkflowSessionWrapper(cqWorkflowService, workflowSession);
                CQWorkItemWrapper workItemWrapper = new CQWorkItemWrapper(workItem);
                CQMetaDataMap cqMetaDataMap = new CQMetaDataMap(metaDataMap);
                String args[] = getArgs(workItem);

                if (toBeExecuted instanceof WorkflowProcess) {
                    WorkflowProcess process = (WorkflowProcess) toBeExecuted;

                    process.execute(workItemWrapper, sessionWrapper, cqMetaDataMap);
                }
                else if (toBeExecuted instanceof JavaProcessExt) {
                    JavaProcessExt process = (JavaProcessExt) toBeExecuted;
                    process.execute(workItemWrapper, sessionWrapper, args);
                }
                else if (toBeExecuted instanceof JavaProcess) {
                    JavaProcess process = (JavaProcess) toBeExecuted;
                    process.execute(workItemWrapper, sessionWrapper);
                } else {
                    // should actually never happen
                    log.error("unsupported process instance " + toBeExecuted);
                }
            } catch (Throwable t) {
                log.error("Process execution resulted in an error: " + t.getMessage(), t);
                throw new WorkflowException("Failed to execute process", t);
            }
        }
    }

    public boolean canExecute(String resource) {
        return evaluateExecutable(resource)!=null;
    }

    /**
     * Attempts to evaluate the process implementation for the given resource.
     *
     * @param resource
     *            The name of the process implementation to search for.
     * @return The process implementation to use, or <code>null</code> if none
     *         can be evaluated.
     */
    private Object evaluateExecutable(String resource) {
        Object toBeExecuted = null;
        for (WorkflowProcess workfowProcess : workflowProcesses) {
            if (resource.equals(workfowProcess.getClass().getName())) {
                toBeExecuted = workfowProcess;
                break;
            }
        }

        // TODO remove after 5.4
        if (toBeExecuted == null) {
            for (JavaProcess javaProcess : processes) {
                if (resource.equals(javaProcess.getClass().getName())) {
                    toBeExecuted = javaProcess;
                    break;
                }
            }
        }

        return toBeExecuted;
    }

    protected String[] getArgs(WorkItem item) {
        String arguments = item.getNode().getMetaDataMap().get(CQWorkflowNodeWrapper.PROCESS_ARGS, String.class);
        if (arguments != null && !arguments.equals("")) {
            return arguments.split(",");
        }
        return new String[0];
    }
}
