/*
 * Copyright 1997-2008 Day Management AG
 * Barfuesserplatz 6, 4001 Basel, Switzerland
 * All Rights Reserved.
 *
 * This software is the confidential and proprietary information of
 * Day Management AG, ("Confidential Information"). You shall not
 * disclose such Confidential Information and shall use it only in
 * accordance with the terms of the license agreement you entered into
 * with Day.
 */
package com.day.cq.workflow.exec;

import java.util.Date;
import java.util.List;

import com.day.cq.workflow.HasMetaData;
import com.day.cq.workflow.exec.filter.WorkItemFilter;
import com.day.cq.workflow.model.WorkflowModel;

/**
 * A <code>Workflow</code> represents the current state of an instance of a
 * <code>{@link WorkflowModel}</code>. In terms of petri nets the state of a
 * workflow instance can be considered as the collections of tokens (or
 * <code>{@link WorkItem}</code>s) that flow through the workflow instance.
 */
public interface Workflow extends HasMetaData {
	/**
	 * Returns the <code>Workflow</code> instance ID.
	 * 
	 * @return The <code>Workflow</code> instance ID.
	 */
	String getId();

	/**
	 * Returns the list of <code>{@link WorkItem}</code>s assigned to the
	 * <code>Workflow</code> instance.
	 * 
	 * @return The list of <code>{@link WorkItem}</code>s assigned to the
	 *         <code>Workflow</code>
	 */
	List<WorkItem> getWorkItems();

    /**
	 * Returns a filtered list of <code>{@link WorkItem}</code>s assigned to the
	 * <code>Workflow</code> instance.
	 *
	 * @param filter One is able to filter a subset of the available
     * <code>{@link WorkItem}</code>s by passing a <code>{@link WorkItemFilter}</code> impl.
     *
     * @return The list of <code>{@link WorkItem}</code>s assigned to the
	 *         <code>Workflow</code>
     *
     * @since 5.4
     */
    List<WorkItem> getWorkItems(WorkItemFilter filter);

	/**
	 * Returns the <code>{@link WorkflowModel}</code> that defines the
	 * underlying workflow model of the <code>Workflow</code> instance.
	 * 
	 * @return The <code>{@link WorkflowModel}</code> of the
	 *         <code>Workflow</code>.
	 */
	WorkflowModel getWorkflowModel();

	/**
	 * Indicates if the <code>Workflow</code> instance is still active.
	 * 
	 * @return True if active, otherwise false.
	 */
	boolean isActive();

	/**
	 * Returns the workflows state. In the default implementation this could be
	 * one of <code>RUNNING</code>, <code>SUSPENDED</code>,
	 * <code>COMPLETED</code> or <code>ABORTED</code>.
	 * 
	 * @return The current workflows state.
	 */
	String getState();

	/**
	 * Returns the initiator of the <code>Workflow</code> instance.
	 * 
	 * @return The user who initiated the <code>Workflow</code>.
	 */
	String getInitiator();

	/**
	 * Getter for the point in time where the <code>Workflow</code> instance
	 * was started.
	 * 
	 * @return The start time of the <code>Workflow</code> instance.
	 */
	Date getTimeStarted();

	/**
	 * Getter for the point in time where the <code>Workflow</code> was
	 * finished.
	 * 
	 * @return The end time of the <code>Workflow</code> instance.
	 */
	Date getTimeEnded();

	/**
	 * Returns the {@link WorkflowData} object assigned to the
	 * <code>Workflow</code> instance.
	 * 
	 * @return The {@link WorkflowData} object assigned to the
	 *         <code>Workflow</code> instance.
	 */
	WorkflowData getWorkflowData();
}
