/*
 * Copyright 1997-2008 Day Management AG
 * Barfuesserplatz 6, 4001 Basel, Switzerland
 * All Rights Reserved.
 *
 * This software is the confidential and proprietary information of
 * Day Management AG, ("Confidential Information"). You shall not
 * disclose such Confidential Information and shall use it only in
 * accordance with the terms of the license agreement you entered into
 * with Day.
 */

package com.day.cq.workflow.job;

import com.day.cq.workflow.exec.WorkItem;

import java.io.Serializable;
import java.util.Dictionary;
import java.util.Hashtable;
import java.util.Date;
import java.util.Calendar;

import org.osgi.service.event.Event;
import org.apache.sling.event.EventUtil;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * The <code>TimeoutJob</code> class provides a utility for creating event
 * triggered job used for programmed/scripted workflow steps.
 */
public class TimeoutJob implements Serializable {

    private static final Logger log = LoggerFactory.getLogger(TimeoutJob.class);

	/** Generated serial version UID. */
	private static final long serialVersionUID = 5670996916430565635L;

	/** The job topic for adding an entry to the audit log. */
	public static final String TIMEOUT_JOB_TOPIC = "com/day/cq/workflow/timeout/job";

	/** The event property holding the {@link TimeoutJob}. */
	public static final String TIMEOUT_JOB = "com.day.cq.workflow.timeout.job";

	/** The {@link com.day.cq.workflow.exec.WorkItem} <code>Id</code> of the job. */
	protected String itemId;

    /** The timeout job handler */
    String handler;

    protected long timeStarted = 0;

    /** Creates a new WorkflowJob.
     *
     * @param item {@link WorkItem} of the job to be created
     *
     * @param handler Handler of the job
     *
     * */
	public TimeoutJob(WorkItem item, String handler) {
		if (item == null || handler == null) {
			throw new IllegalArgumentException("item or handler must not be null.");
		}
		this.itemId = item.getId();
        this.timeStarted = item.getTimeStarted().getTime();
        this.handler = handler;
    }

	/** Returns the {@link com.day.cq.workflow.exec.WorkItem} <code>Id</code> of the job
     *
     * @return String work item id
     * */
	public String getWorkItemId() {
		return itemId;
	}

    /** Returns the timeout handler
     *
     * @return String the name of the handler
     * */
    public String getHandler(){
        return handler;
    }

    /** Convenience method to create a job event for the workflow job.
     *
     * @param executeParallel  true or false. Execute job event in parallel
     *
     * @param seconds timeout value in seconds
     *
     * @param addOffset boolean. add current date/time as starting offset
     *
     * @return Event {@link Event} object created
     * */
	public Event createEvent(boolean executeParallel, long seconds, boolean addOffset) {
		final Dictionary<String, Object> props = new Hashtable<String, Object>();
		props.put(TIMEOUT_JOB, this);
		props.put(EventUtil.PROPERTY_JOB_PARALLEL, executeParallel);
        props.put(EventUtil.PROPERTY_TIMED_EVENT_TOPIC, EventUtil.TOPIC_JOB);
        props.put(EventUtil.PROPERTY_JOB_TOPIC, TIMEOUT_JOB_TOPIC);
        props.put(EventUtil.PROPERTY_JOB_ID, itemId + "_" + timeStarted);
        Date date = calculateDate(seconds, addOffset);
        props.put(EventUtil.PROPERTY_TIMED_EVENT_DATE, date);
        log.debug("create timeout event {} ", props);
        return new Event(EventUtil.TOPIC_TIMED_EVENT, props);
	}

    /** Convenience method to create a job cancel event for the workflow job.
     *
     * @param executeParallel  true or false. Execute job event in parallel
     *
     * @return Event {@link Event} object created
     * */
    public Event cancelEvent(boolean executeParallel) {
		final Dictionary<String, Object> props = new Hashtable<String, Object>();
		props.put(TIMEOUT_JOB, this);
		props.put(EventUtil.PROPERTY_JOB_PARALLEL, executeParallel);
        props.put(EventUtil.PROPERTY_TIMED_EVENT_TOPIC, EventUtil.TOPIC_JOB);
        props.put(EventUtil.PROPERTY_JOB_TOPIC, TIMEOUT_JOB_TOPIC);
        props.put(EventUtil.PROPERTY_JOB_ID, itemId + "_" + timeStarted);
        log.debug("create timeout cancel event {}", props);
        return new Event(EventUtil.TOPIC_TIMED_EVENT, props);
	}

    private Date calculateDate(long seconds, boolean addOffset) {
        Calendar cal = Calendar.getInstance();
        long millies = seconds * 1000;
        if (addOffset) {
            millies += cal.getTimeInMillis();
        }
        cal.setTimeInMillis(millies);
        return cal.getTime();
    }
}
