/*************************************************************************
*
* ADOBE CONFIDENTIAL
* ___________________
*
*  Copyright 1997 Adobe Systems Incorporated
*  All Rights Reserved.
*
* NOTICE:  All information contained herein is, and remains
* the property of Adobe Systems Incorporated and its suppliers,
* if any.  The intellectual and technical concepts contained
* herein are proprietary to Adobe Systems Incorporated and its
* suppliers and are protected by trade secret or copyright law.
* Dissemination of this information or reproduction of this material
* is strictly forbidden unless prior written permission is obtained
* from Adobe Systems Incorporated.
**************************************************************************/
package com.day.crx.statistics;

import org.apache.jackrabbit.commons.JcrUtils;

import javax.jcr.Session;
import javax.jcr.RepositoryException;
import javax.jcr.Node;

/**
 * <code>Entry</code> is the basic unit of statistic information that can be
 * stored in a statistics workspace.
 *
 * @author mreutegg
 */
public abstract class Entry {

    /**
     * Timestamp for this entry.
     */
    private long timestamp = System.currentTimeMillis();

    /**
     * The path prefix.
     */
    private String pathPrefix;

    /**
     * Creates a new entry.
     *
     * @param pathPrefix the location where the entry will be stored.
     */
    public Entry(String pathPrefix) {
        setPathPrefix(pathPrefix);
    }

    /**
     * Writes this entry to the workspace accessible via <code>session</code>.
     *
     * @param session the session giving access to the worksapce.
     * @throws RepositoryException if an error occurs while writing to the
     *                             workspace.
     */
    public final void write(Session session) throws RepositoryException {
        String path = getPath();
        Node node = JcrUtils.getOrCreateByPath(path, "nt:unstructured", "nt:unstructured", session, true);
        write(node);
    }

    /**
     * Writes the timestamp to a long property called <code>timestamp</code>.
     *
     * @param node the node where to store the property.
     * @throws RepositoryException if an error occurs while writing to the node.
     */
    public void write(Node node) throws RepositoryException {
        node.setProperty("timestamp", getTimestamp());
    }

    /**
     * @return the path where this entry is stored in the workspace.
     */
    public final String getPath() {
        StringBuffer buffer = new StringBuffer();
        getPathBuilder().formatPath(this, buffer);
        return buffer.toString();
    }

    /**
     * @return a path builder for this kind of entry.
     */
    protected abstract PathBuilder getPathBuilder();

    //-------------------------------< properties >-----------------------------

    /**
     * @return a timestamp for this entry.
     */
    public long getTimestamp() {
        return timestamp;
    }

    /**
     * @param timestamp a time stamp for this entry.
     */
    public void setTimestamp(long timestamp) {
        this.timestamp = timestamp;
    }

    /**
     * Returns an absolute path prefix or an empty string if no path prefix is
     * set. This method will never return the path of the root node ('/'),
     * instead it will return an empty string.
     *
     * @return an absolute path prefix or an empty string.
     */
    public String getPathPrefix() {
        return pathPrefix;
    }

    /**
     * @param pathPrefix location where the entry will be stored.
     */
    public void setPathPrefix(String pathPrefix) {
        // make sure path is absolute
        if (!pathPrefix.startsWith("/")) {
            pathPrefix = "/" + pathPrefix;
        }
        if (pathPrefix.equals("/")) {
            pathPrefix = "";
        }
        this.pathPrefix = pathPrefix;
    }
}
