/*************************************************************************
*
* ADOBE CONFIDENTIAL
* ___________________
*
*  Copyright 1997 Adobe Systems Incorporated
*  All Rights Reserved.
*
* NOTICE:  All information contained herein is, and remains
* the property of Adobe Systems Incorporated and its suppliers,
* if any.  The intellectual and technical concepts contained
* herein are proprietary to Adobe Systems Incorporated and its
* suppliers and are protected by trade secret or copyright law.
* Dissemination of this information or reproduction of this material
* is strictly forbidden unless prior written permission is obtained
* from Adobe Systems Incorporated.
**************************************************************************/
package com.day.crx.statistics.query;

import com.day.crx.statistics.JanitorReport;

import javax.jcr.Session;
import javax.jcr.RepositoryException;
import javax.jcr.Node;
import javax.jcr.NodeIterator;
import java.util.Iterator;
import java.util.SortedSet;
import java.util.TreeSet;
import java.util.Calendar;
import java.util.List;
import java.util.ArrayList;
import java.util.Collections;

/**
 * <code>RemoveOldQueriesReport</code> implements a report, which removes old
 * query statistics based on two threshold dates, one for monthly and another
 * one for daily statistics.
 *
 * @author mreutegg
 */
public class RemoveOldQueriesReport extends JanitorReport {

    /**
     * Creates a new report.
     *
     * @param dataPath the path where query data is stored.
     * @param dryRun   whether the data should actually be removed.
     */
    public RemoveOldQueriesReport(String dataPath, boolean dryRun) {
        super(dataPath, dryRun);
    }

    /**
     * {@inheritDoc}
     * <p/>
     * Returns result rows with the following objects:
     * <ul>
     * <li>Path <code>String</code> of the node that was removed.</li>
     * </ul>
     */
    public Iterator getResult(Session session) throws RepositoryException {
        SortedSet removable = new TreeSet(Collections.reverseOrder());
        Calendar cal = Calendar.getInstance();
        Node data = getDataNode(session);
        if (data == null) {
            // no query data
            return Collections.EMPTY_LIST.iterator();
        }
        for (NodeIterator years = data.getNodes(); years.hasNext(); ) {
            Node y = years.nextNode();
            cal.set(Calendar.YEAR, Integer.parseInt(y.getName()));
            for (NodeIterator months = y.getNodes(); months.hasNext(); ) {
                Node m = months.nextNode();
                // month is 0-based
                cal.set(Calendar.MONTH, (Integer.parseInt(m.getName()) - 1));
                cal.set(Calendar.DAY_OF_MONTH, 1);
                if (cal.getTimeInMillis() < getMonthThresholdDate()) {
                    removable.add(m.getPath());
                } else {
                    // check days
                    for (NodeIterator days = m.getNodes(); days.hasNext(); ) {
                        Node d = days.nextNode();
                        cal.set(Calendar.DAY_OF_MONTH, Integer.parseInt(d.getName()) - 1);
                        if (cal.getTimeInMillis() < getDayThresholdDate()) {
                            removable.add(d.getPath());
                        }
                    }
                }
            }
        }
        List result = new ArrayList();
        for (Iterator it = removable.iterator(); it.hasNext(); ) {
            String path = (String) it.next();
            if (!isDryRun()) {
                session.getItem(path).remove();
            }
            result.add(new Object[]{path});
        }
        if (!isDryRun()) {
            session.save();
        }
        return result.iterator();
    }
}
