/*************************************************************************
*
* ADOBE CONFIDENTIAL
* ___________________
*
*  Copyright 1997 Adobe Systems Incorporated
*  All Rights Reserved.
*
* NOTICE:  All information contained herein is, and remains
* the property of Adobe Systems Incorporated and its suppliers,
* if any.  The intellectual and technical concepts contained
* herein are proprietary to Adobe Systems Incorporated and its
* suppliers and are protected by trade secret or copyright law.
* Dissemination of this information or reproduction of this material
* is strictly forbidden unless prior written permission is obtained
* from Adobe Systems Incorporated.
**************************************************************************/
package com.day.crx.statistics.result;

import com.day.crx.statistics.Report;

import javax.jcr.Session;
import javax.jcr.RepositoryException;
import javax.jcr.NodeIterator;
import javax.jcr.Node;
import javax.jcr.query.QueryManager;
import javax.jcr.query.Query;
import java.util.Iterator;
import java.util.List;
import java.util.ArrayList;
import java.util.Map;
import java.util.HashMap;
import java.util.Collections;
import java.util.Comparator;

/**
 * <code>ResultsByQueryReport</code> implements a report, which returns the
 * result pages that were selected the most for a given query. This report
 * operates on the daily statistics nodes.
 *
 * @author mreutegg
 */
public class ResultsByQueryReport extends Report {

    /**
     * The user query.
     */
    private final String query;

    /**
     * Number of results to return at most. Default value: top 10 results.
     */
    private int size = 10;

    /**
     * Add OAK-specific "option(traversal ok)" to the statistics query so that no traversal warning is triggered
     */
    private final boolean traversalOk;

    /**
     * Creates a new report.
     *
     * @param dataPath the path prefix to the data.
     * @param query    the user query.
     */
    public ResultsByQueryReport(String dataPath, String query) {
        this(dataPath, query, false);
    }

    /**
     * Creates a new report.
     *
     * @param dataPath the path prefix to the data.
     * @param query the user query.
     * @param traversalOk set to true to put "traveral ok" option in the statistics query
     */
    public ResultsByQueryReport(String dataPath, String query, boolean traversalOk) {
        super(dataPath);
        this.query = query;
        this.traversalOk = traversalOk;
    }

    /**
     * {@inheritDoc}
     * <p/>
     * Returns result rows with the following objects:
     * <ul>
     * <li><code>String</code> path of a result path</li>
     * <li><code>Long</code> count (how many times this result was picked)</li>
     * </ul>
     */
    public Iterator getResult(Session session) throws RepositoryException {
        QueryManager qm = session.getWorkspace().getQueryManager();
        StringBuffer stmt = new StringBuffer("/jcr:root");
        stmt.append(getDataPath());
        stmt.append("//*[@queries = '");
        stmt.append(query.replaceAll("'", "''"));
        // only day nodes have an avgPosition
        stmt.append("' and @").append(ResultSelected.AVG_POSITION);
        stmt.append("]");
        if (traversalOk) {
            stmt.append(" option(traversal ok)");
        }
        Map paths = new HashMap();
        NodeIterator nodes = qm.createQuery(stmt.toString(),
                Query.XPATH).execute().getNodes();
        while (nodes.hasNext()) {
            Node n = nodes.nextNode();
            String path = n.getPath();
            int idx = path.indexOf(ResultSelectedPathBuilder.STATS_NAME);
            if (idx == -1) {
                continue;
            }
            path = path.substring(0, idx - 1);
            // cut off data path prefix
            path = path.substring(getDataPath().length());
            Long count = (Long) paths.get(path);
            if (count == null) {
                count = new Long(0);
            }
            if (n.hasProperty(ResultSelected.COUNT)) {
                count = new Long(n.getProperty(ResultSelected.COUNT).getLong() +
                        count.longValue());
                paths.put(path, count);
            }
        }
        List data = new ArrayList();
        for (Iterator it = paths.entrySet().iterator(); it.hasNext(); ) {
            Map.Entry entry = (Map.Entry) it.next();
            data.add(new Object[]{entry.getKey(), entry.getValue()});
        }
        Collections.sort(data, new Comparator() {
            public int compare(Object o1, Object o2) {
                Object[] e1 = (Object[]) o1;
                Object[] e2 = (Object[]) o2;
                int comp = ((Comparable) e2[1]).compareTo(e1[1]);
                if (comp != 0) {
                    return comp;
                }
                return ((String) e1[0]).toLowerCase().compareTo(((String) e2[0]).toLowerCase());
            }
        });
        data = data.subList(0, Math.min(getSize(), data.size()));
        return data.iterator();
    }

    //-----------------------------< properties >-------------------------------

    /**
     * @return the maximum number of results to return.
     */
    public int getSize() {
        return size;
    }

    /**
     * @param size the maximum number of results to return.
     */
    public void setSize(int size) {
        this.size = size;
    }
}

