/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * __________________
 *
 *  Copyright 2012 Adobe Systems Incorporated
 *  All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Systems Incorporated and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Systems Incorporated.
 **************************************************************************/
package com.day.image.font;

import java.awt.Paint;
import java.awt.Stroke;
import java.awt.geom.Rectangle2D;
import java.text.CharacterIterator;

import com.day.image.Layer;
import com.day.image.internal.font.FontHelper;


/**
 * The <code>AbstractFont</code> class provides base class for font extensions
 * with some static helper methods and access to the {@link FontHelper}
 * and the system {@link Ticket} to access the ContentBus to its extensions.
 * <p>
 * This class cannot be initialized if the {@link FontHelper} and/or the
 * system {@link Ticket} are not available.
 *
 * @version $Revision: 20173 $, $Date: 2006-05-03 16:47:57 +0200 (Mit, 03 Mai 2006) $
 * @author fmeschbe
 * @since degu
 * @audience core
 */
public abstract class AbstractFont {

    //---------- public constants ----------------------------------------------

    /** Request plain font style (default) */
    public static final int PLAIN = java.awt.Font.PLAIN;

    /** Request bold font style */
    public static final int BOLD = java.awt.Font.BOLD;

    /** Request italic font style */
    public static final int ITALIC = java.awt.Font.ITALIC;

    /**
     * The underline style constant. This can be combined with the other style
     * constants (except PLAIN) for mixed styles.
     * <p>
     * Underlining in Java2D is a property of drawing text and not a font
     * style. Thus this constant is valid for any font you choose and takes
     * effect when drawing text - and it works.
     */
    public static final int UNDERLINE = 4;

    /**
     * The strikeout style constant. This can be combined with the other style
     * constants (except PLAIN) for mixed styles.
     * <p>
     * Striekout in Java2D is a property of drawing text and not a font
     * style. Thus this constant is valid for any font you choose and takes
     * effect when drawing text - and it works.
     */
    public static final int STRIKEOUT = 8;

    /**
     * The old fonter scaling constant. Set this constant to change the scaling
     * of fonts from the old 72dpi to the default 96dpi.
     */
    public static final int TTFONTERSCALE = 0x100;

    /**
     * The old line spacing constant. Set this constant to use the new line
     * spacing values instead of the old line spacing values, which tend
     * to be somewhat bigger, than what is intended in the TrueType standard.
     */
    public static final int TTFONTERLINESPACING = 0x200;

    /** The vertical alignment flags mask. */
    public static final int ALIGN_VBASE = 0x0f;

    /**
     * The top alignment constant. This can be combined with any one of the
     * horizontal alignment and rotation constants to get mixed alignment/rotation
     * style.
     */
    public static final int ALIGN_TOP = 0x00;

    /**
     * The base line alignment constant. This can be combined with any one of the
     * horizontal alignment and rotation constants to get mixed alignment/rotation
     * style.
     */
    public static final int ALIGN_BASE = 0x01;

    /**
     * The bottom alignment constant. This can be combined with any one of the
     * horizontal alignment and rotation constants to get mixed alignment/rotation
     * style.
     */
    public static final int ALIGN_BOTTOM = 0x02;

    /** The vertical alignment flags mask. */
    public static final int ALIGN_HBASE = 0xf0;

    /**
     * The left alignment constant. This can be combined with any one of the
     * vertical alignment and rotation constants to get mixed alignment/rotation
     * style.
     */
    public static final int ALIGN_LEFT = 0x00;

    /**
     * The center alignment constant. This can be combined with any one of the
     * vertical alignment and rotation constants to get mixed alignment/rotation
     * style.
     */
    public static final int ALIGN_CENTER = 0x10;

    /**
     * The right alignment constant. This can be combined with any one of the
     * vertical alignment and rotation constants to get mixed alignment/rotation
     * style.
     */
    public static final int ALIGN_RIGHT = 0x20;

    /**
     * The full justification alignment constant. This can be combined with any
     * one of the vertical alignment and rotation constants to get mixed
     * alignment/rotation style.
     */
    public static final int ALIGN_FULL = 0x40;

    /** The rotation flags mask. */
    public static final int ROTBASE = 0x0f00;

    /**
     * The odd (?) rotation constant. This can be combined with any one of the
     * vertical and horizontal alignment constants to get mixed alignment/rotation
     * style.
     */
    public static final int ROTODD = 0x0300;

    /**
     * The 90� (clockwise) rotation constant. This can be combined with any one of the
     * vertical and horizontal alignment constants to get mixed alignment/rotation
     * style.
     */
    public static final int ROT90 = 0x0100;

    /**
     * The 270� (clockwise) rotation constant. This can be combined with any one of the
     * vertical and horizontal alignment constants to get mixed alignment/rotation
     * style.
     */
    public static final int ROT270 = 0x0200;

    /**
     * The 180� (clockwise) rotation constant. This can be combined with any one of the
     * vertical and horizontal alignment constants to get mixed alignment/rotation
     * style.
     */
    public static final int ROT180 = 0x0400;

    /** The text drawing style flags. */
    public static final int DRAWBASE = 0xf000;

    /**
     * The underline style constant. This can be combined with the other style
     * constants (except PLAIN) for mixed styles.
     * <p>
     * Underlining in Java2D is a property of drawing text and not a font
     * style. Thus this constant is valid for any font you choose and takes
     * effect when drawing text - and it works.
     */
    public static final int DRAW_UNDERLINE = 0x1000;

    /**
     * The strikeout style constant. This can be combined with the other style
     * constants (except PLAIN) for mixed styles.
     * <p>
     * Striekout in Java2D is a property of drawing text and not a font
     * style. Thus this constant is valid for any font you choose and takes
     * effect when drawing text - and it works.
     */
    public static final int DRAW_STRIKEOUT = 0x2000;

    /**
     * The draw the character outline constant instructs the font renderer to
     * draw the texts outline and not fill it. Default is filling the outline.
     * This can be combined with any one of the vertical and horizontal
     * alignment and rotation constants to get mixed alignment/rotation style.
     */
    public static final int DRAW_OUTLINE = 0x4000;

    /** The TrueType flags mask */
    public static final int TTBASE = 0x0f0000;

    /**
     * The antialiased font rendering constant. This can be combined with any one of the
     * vertical and horizontal alignment and rotation constants to get mixed
     * alignment/rotation style.
     */
    public static final int TTANTIALIASED = 0x010000;

    /**
     * The hinting font rendering constant. This can be combined with any one of the
     * vertical and horizontal alignment and rotation constants to get mixed
     * alignment/rotation style.
     */
    public static final int TTHINTED = 0x020000;

    /**
     * The oversampling font rendering constant. This can be combined with any
     * one of the vertical and horizontal alignment and rotation constants to
     * get mixed alignment/rotation style.
     * <p>
     * By popular demand font oversampling has been added again. For this reason
     * this flag is not deprecated any more.
     * <p>
     * The factor applied to the text being drawn is set by the
     * {@link #setOversamplingFactor(int)} method and has a default value of
     * 16.
     *
     * @see #drawText(Layer, int, int, int, int, String, Paint, Stroke, int, double, int)
     * @see #getOversamplingFactor()
     * @see #setOversamplingFactor(int)
     */
    public static final int TTOVERSAMPLING = 0x040000;

    /**
     * The 'do-not-hint' font rendering constant. This can be combined with any one of the
     * vertical and horizontal alignment and rotation constants to get mixed
     * alignment/rotation style.
     *
     * @deprecated since cq3
     */
    @Deprecated
    public static final int TTUNHINTED = 0x080000;

    //--------- helper methods -------------------------------------------------

    /**
     * Constructs and returns a font file name from the font face name, size
     * and style indication. The file name consists of the face name converted
     * to lower case and blanks replaced by underscores appending the size
     * and style converted to a string with {@link #styleToString}.
     * <p>
     * Example : The font <em>Times New Romain</em>, size 12, style bold/italic
     * has the font file name <em>times_new_roman12bi</em>.
     *
     * @param faceName The font face name
     * @param size The size of the font
     * @param style The style flags of the fon
     *
     * @return The valid font file name
     */
    public static String createFontFileName(String faceName, int size, int style) {
        // convert to lower case replacing invalid characters with _
        StringBuffer buf = new StringBuffer();
        for (int i=0; i < faceName.length(); i++) {
            char c = faceName.charAt(i);

            // map to lower case or convert to _ if not valid label
            if (Character.isLetterOrDigit(c)) {
                buf.append(Character.toLowerCase(c));
            } else {
                buf.append('_');
            }
        }

        // append size and style
        buf.append(size);
        buf.append(styleToString(style));

        return buf.toString();
    }

    /**
     * Converts the numeric font style code to a descriptive string.
     *
     * @param style The numeric font style
     *
     * @return The descriptive string representation of the font style
     */
    public static String styleToDescription(int style) {
        if (style == 0) {
            return "Plain";
        }

        StringBuilder buf = new StringBuilder();
        if ((style & BOLD) != 0) buf.append("Bold ");
        if ((style & ITALIC) != 0) buf.append("Italic ");
        if ((style & UNDERLINE) != 0) buf.append("Underline ");
        if ((style & STRIKEOUT) != 0) buf.append("Strikeout ");
        return buf.toString().trim();
    }

    /**
     * Converts the numeric font style code to a string usable for the
     * font name identification.
     *
     * @param style The numeric font style
     *
     * @return The string representation of the font style
     */
    public static String styleToString(int style) {
        char[] st = new char[4];
        int i = 0;

        if ((style & BOLD) != 0) st[i++] = 'b';
        if ((style & ITALIC) != 0) st[i++] = 'i';
        if ((style & UNDERLINE) != 0) st[i++] = 'u';
        if ((style & STRIKEOUT) != 0) st[i++] = 's';

        return String.valueOf(st, 0, i);
    }

    /**
     * Converts the font style code to a string.
     *
     * @param style The string representation of the font style
     *
     * @return The numeric font style code.
     */
    public static int stringToStyle(String style) {
        int type = 0;

        if (style.indexOf('b') >= 0) type |= BOLD;
        if (style.indexOf('i') >= 0) type |= ITALIC;
        if (style.indexOf('u') >= 0) type |= UNDERLINE;
        if (style.indexOf('s') >= 0) type |= STRIKEOUT;

        return type;
    }

    //---------- AbstractFont API

    /**
     * Calculate the bounding box of the text, if it would be rendered with the
     * rendering attributes given. The calculation will be done as if the text
     * would be rendered in the current font object.
     * <p>
     * If the width of the text box is set to some value other than zero, the
     * text is broken to fit lines of the given length. The line breaking
     * algorithm breaking algorithm breaks on whitespace (blank, tab), carriage
     * return and linefeed (CR/LF) in any combination as well as on other
     * characters such as hyphens.
     * <p>
     * If the text contains carriage return and/or linefeed characters, the
     * text is broken into several lines, regardless of whether the text would
     * be broken because of the text box width setting.
     *
     * @param x left edge of the text box, required
     * @param y top edge of the text box, required
     * @param width maximum width of the textbox. If 0 (or negative) the width
     *              of the bounding box is dependent of the rendering attributes
     * @param height maximum height of the textbox. If 0 (or negative) the width
     *              of the bounding box is dependent of the rendering attributes
     * @param text the text string to calculate the bounding box for
     * @param align alignment, rotation and TrueType attributes for the text.
     *              Use {@link #ALIGN_LEFT} as default value.
     * @param cs extra intercharacter spacing.
     *              Use 0 as default value to not add additional space.
     * @param ls extra line spacing.
     *              Use 0 as default value to not add additional space.
     *
     * @return returns a rectangle representing the bounding box, if the text
     *         would be rendered in this font using the given additional
     *         rendering attributes.
     */
    public abstract Rectangle2D getTextExtent(int x, int y, int width, int height,
            String text, int align, double cs, int ls);

    /**
     * Render the given text string in the given font to the <code>Layer</code>
     * using the attributes given. Use the default values given for unspecified
     * values.
     * <p>
     * If the width of the text box is set to some value other than zero, the
     * text is broken to fit lines of the given length. The line breaking
     * algorithm breaking algorithm breaks on whitespace (blank, tab), carriage
     * return and linefeed (CR/LF) in any combination as well as on other
     * characters such as hyphens.
     * <p>
     * If the text contains carriage return and/or linefeed characters, the
     * text is broken into several lines, regardless of whether the text would
     * be broken because of the text box width setting.
     *
     * @param layer the layer to draw the text into
     * @param x left edge of the text box, required
     * @param y top edge of the text box, required
     * @param width maximum width of the textbox. If 0 (or negative) the width
     *              of the bounding box is dependent of the rendering attributes
     * @param height maximum height of the textbox. If 0 (or negative) the width
     *              of the bounding box is dependent of the rendering attributes
     * @param text the text string to calculate the bounding box for
     * @param paint The <code>Paint</code> to use for the text drawing.
     * @param stroke The <code>Stroke</code> to use for the text drawing.
     * @param align alignment, rotation and TrueType attributes for the text.
     *              Use {@link Font#ALIGN_LEFT} as default value.
     * @param cs extra intercharacter spacing.
     *              Use 0 as default value to not add additional space.
     * @param ls extra line spacing.
     *              Use 0 as default value to not add additional space.
     *
     * @return the number of text lines drawn to the layer
     */
    public abstract int drawText(Layer layer, int x, int y, int width, int height,
            String text, Paint paint, Stroke stroke, int align, double cs, int ls);

    /**
     * Returns the calculated font height in pixels.
     * @return the calculated font height in pixels.
     */
    public abstract double getHeight();

    /**
     * Returns the ascent of this <code>Font</code> object, which is the maximal
     * value any glyph of this font ascends above the base line.
     *
     * @return The ascent of this <code>Font</code>.
     */
    public abstract double getAscent();

    /**
     * Returns the descent of this <code>Font</code> object, which is the
     * maximal value any glyph of this font descends below the base line.
     *
     * @return The ascent of this <code>Font</code>.
     */
    public abstract double getDescent();

    /**
     * Checks if this Font has a glyph for the specified character.
     *
     * @param c a unicode character code
     *
     * @return <code>true</code> if this Font can display the character;
     *      <code>false</code> otherwise.
     */
    public abstract boolean canDisplay(char c);

    /**
     * Indicates whether or not this Font can display the specified String. For
     * strings with Unicode encoding, it is important to know if a particular
     * font can display the string. This method returns an offset into the
     * String str which is the first character this Font cannot display without
     * using the missing glyph code . If this Font can display all characters,
     * -1 is returned.
     *
     * @param iter a CharacterIterator object
     * @param start the specified starting offset into the specified array of
     *      characters
     * @param limit the specified ending offset into the specified array of
     *      characters
     *
     * @return an offset into the String object that can be displayed by this
     *      Font.
     */
    public abstract int canDisplayUpTo(CharacterIterator iter, int start, int limit);

    /**
     * Indicates whether or not this Font can display a specified String. For
     * strings with Unicode encoding, it is important to know if a particular
     * font can display the string. This method returns an offset into the
     * String str which is the first character this Font cannot display without
     * using the missing glyph code. If the Font can display all characters,
     * -1 is returned.
     *
     * @param str a String object
     *
     * @return an offset into str that points to the first character in str
     *      that this Font cannot display; or -1 if this Font can display all
     *      characters in str.
     */
    public abstract int canDisplayUpTo(String str);

    /**
     * Indicates whether or not this Font can display the characters in the
     * specified text starting at start and ending at limit. This method is a
     * convenience overload.
     *
     * @param text the specified array of characters
     * @param start the specified starting offset into the specified array of
     *      characters
     * @param limit the specified ending offset into the specified array of
     *      characters
     *
     * @return an offset into text that points to the first character in text
     *      that this Font cannot display; or -1 if this Font can display all
     *      characters in text.
     */
    public abstract int canDisplayUpTo(char[] text, int start, int limit);

    /**
     * Returns a <code>String</code> representation of this object containing
     * the font name, the size and style flags.
     * @return A <code>String</code> representation of this object.
     */
    public abstract String toString();

    public abstract java.awt.Font getAwtFont();

}
