/*************************************************************************
 * ADOBE CONFIDENTIAL
 * __________________
 *
 *  Copyright 2020 Adobe
 *  All Rights Reserved.
 *
 * NOTICE: All information contained herein is, and remains
 * the property of Adobe and its suppliers, if any. The intellectual
 * and technical concepts contained herein are proprietary to Adobe
 * and its suppliers and are protected by all applicable intellectual
 * property laws, including trade secret and copyright laws.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe.
 **************************************************************************/
package com.day.util;

import java.util.*;
import java.text.DecimalFormat;

/**
 * The <code>TimeZoneUtil</code> class provides convenience methods for handling
 * time zones in Java.
 *
 * @version $Revision: 1.3 $, $Date: 2004-08-22 06:56:09 +0200 (Sun, 22 Aug 2004) $
 * @author mreutegg
 * @since echidna
 * Audience wad
 */
public class TimeZoneUtil {

    private static TimeZone[] zones = null;

    private static final DecimalFormat offsetFormatter = new DecimalFormat("00.00");

    /**
     * Returns an array of <code>TimeZone</code> objects sorted in ascending order
     * by their UTC time offset.
     *
     * @return a sorted array of <code>TimeZone</code> objects.
     */
    public synchronized static TimeZone[] getAvailableZones() {
        if (zones == null) {
            String[] allZones = TimeZone.getAvailableIDs();
            // fill a Vector and ignore all three letter id's, but keep UTC
            Vector uniqueZones = new Vector();
            for (int i=0; i < allZones.length; i++) {
                // ignore all three letter id's, but keep UTC
                if (allZones[i].length() <= 3 && !allZones[i].equals("UTC")) continue;
                TimeZone zone = TimeZone.getTimeZone(allZones[i]);
                uniqueZones.add(zone);
            }
            zones = new TimeZone[0];
            zones = (TimeZone[])uniqueZones.toArray(zones);
            Arrays.sort(zones, new TimeZoneComparator());
        }
        // return a copy of the internal array
        TimeZone[] retVal = new TimeZone[zones.length];
        System.arraycopy(zones, 0, retVal, 0, zones.length);

        return retVal;
    }

    /**
     * Returns the <code>TimeZone</code> of the Communique server.
     * @return the <code>TimeZone</code> of the Communique server.
     */
    public static TimeZone getServerTimeZone() {
        return TimeZone.getDefault();
    }

    /**
     * Formats the passed <code>TimeZone</code> object to a String representation:
     * <p><code>[&lt;UTC offset&gt;] &lt;time zone name&gt; (&lt;short name&gt;)</code>
     * <p>Example: [UTC+01.00] Europe/Berlin (CEST)

     * @param zone The <code>TimeZone</code> to format
     * @return a <code>String</code> representation of the passed <code>TimeZone</code>.
     */
    public static String getDisplayName(TimeZone zone) {
        StringBuffer buffer = new StringBuffer("[UTC");
        if (zone.getRawOffset() >= 0) buffer.append("+");
        buffer.append(offsetFormatter.format(((double)zone.getRawOffset())/(1000*60*60)));
        buffer.append("] ").append(zone.getID());
        buffer.append(" (").append(zone.getDisplayName(true, TimeZone.SHORT));
        buffer.append(")");
        return buffer.toString();
    }

    /**
     * Inner utility class for comparing two TimeZone objects regarding their UTC
     * offset.
     */
    static class TimeZoneComparator implements Comparator {

        /**
         * Compares its two arguments (TimeZone objects) for order.
         *
         * @param o1 the first object to be compared.
         * @param o2 the second object to be compared.
         * @return a negative integer, zero, or a positive integer as the
         * 	       first argument is less than, equal to, or greater than the
         *	       second.
         * @throws ClassCastException if the arguments' types prevent them from
         * 	       being compared by this Comparator.
         */
        public int compare(Object o1, Object o2) {
            int diff = ((TimeZone)o1).getRawOffset() - ((TimeZone)o2).getRawOffset();
            return (diff != 0) ? diff : ((TimeZone)o1).getID().compareTo(((TimeZone)o2).getID());
        }
    }
}
