/*
 *
 * ADOBE CONFIDENTIAL
 * __________________
 *
 *  Copyright 2017 Adobe Systems Incorporated
 *  All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Systems Incorporated and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Systems Incorporated.
 */
package com.day.util.diff;

import java.io.PrintWriter;

/**
 * Provides a default output for a diff.
 */
public class DefaultChangeListener implements ChangeListener {

    /**
     * the output writer
     */
    private final PrintWriter out;

    /**
     * debug flag
     */
    private boolean debug;

    /**
     * Creates a new default change listener that will write to the given
     * writer.
     * @param out the writer
     */
    public DefaultChangeListener(PrintWriter out) {
        this.out = out;
    }

    /**
     * Creates a new default change listener that will write to the given
     * writer. if debug is <code>true</code> the line numbers are also included
     * in the output.
     *
     * @param out the writer
     * @param debug flag
     */
    public DefaultChangeListener(PrintWriter out, boolean debug) {
        this.out = out;
        this.debug = debug;
    }

    /**
     * {@inheritDoc}
     */
    public void onDocumentsStart(Document left, Document right) {
        out.println("Start Diff");
    }

    /**
     * {@inheritDoc}
     */
    public void onDocumentsEnd(Document left, Document right) {
        out.println("End Diff");
    }

    /**
     * {@inheritDoc}
     */
    public void onChangeStart(int leftLine, int leftLen, int rightLine, int rightLen) {
        out.println("@@ -" + (leftLine+1) + "," + leftLen + " +" + (rightLine+1) + "," + rightLen + " @@");
    }

    /**
     * {@inheritDoc}
     */
    public void onChangeEnd() {
        out.flush();
    }

    /**
     * {@inheritDoc}
     */
    public void onUnmodified(int leftLine, int rightLine, Document.Element text) {
        if (debug) {
            out.print("(" + (leftLine+1) + "," + (rightLine+1) + ") ");
        }
        out.println(text);
    }

    /**
     * {@inheritDoc}
     */
    public void onDeleted(int leftLine, int rightLine, Document.Element text) {
        if (debug) {
            out.print("(" + (leftLine+1) + "," + (rightLine+1) + ") ");
        }
        out.print("-");
        out.println(text);
    }

    /**
     * {@inheritDoc}
     */
    public void onInserted(int leftLine, int rightLine, Document.Element text) {
        if (debug) {
            out.print("(" + (leftLine+1) + "," + (rightLine+1) + ") ");
        }
        out.print("+");
        out.println(text);
    }

}
