/*
 *
 * ADOBE CONFIDENTIAL
 * __________________
 *
 *  Copyright 2017 Adobe Systems Incorporated
 *  All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Systems Incorporated and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Systems Incorporated.
 */
package com.day.util.diff;

/**
 * A document represents a list of elements and have a source.
 */
public class Document {

    /**
     * the source information of this document
     */
    private final DocumentSource source;

    /**
     * the array of elements that form this document
     */
    private final Element[] elements;

    /**
     * Create a new document with the given source and element factory
     * @param source the source
     * @param factory the element factory
     */
    public Document(DocumentSource source, ElementsFactory factory) {
        this.source = source;
        this.elements = factory.getElements();
    }

    /**
     * Return the source of this document
     * @return the source.
     */
    public DocumentSource getSource() {
        return source;
    }

    /**
     * Return the elements of this document
     * @return the elements.
     */
    public Element[] getElements() {
        return elements;
    }

    /**
     * Create a <em>diff</em> between this document and the given one.
     *
     * @param right the other document to diff to.
     * @return a diff.
     */
    public DocumentDiff diff(Document right) {
        return new DocumentDiff(this, right);
    }

    /**
     * Create a <em>diff</em> between the given document and this one.
     * @param left the other document.
     * @return a diff
     */
    public DocumentDiff reverseDiff(Document left) {
        return new DocumentDiff(left, this);
    }

    /**
     * Create a <em>tree-way-diff</em> using this document as base.
     *
     * @param left the left document
     * @param right the right document
     * @return a diff3
     */
    public DocumentDiff3 diff3(Document left, Document right) {
        return new DocumentDiff3(this, left, right);
    }

    /**
     * Elements form a document.
     */
    public static interface Element {

        /**
         * Returns the string representation of this element. If the elements
         * were generated originally from a string they should return the
         * exact string again.
         * @return the string of this element.
         */
        String getString();
    }

    /**
     * The annotated element include the document source. This can be used
     * to create an annotated document. 
     */
    public static interface AnnotatedElement extends Element {

        /**
         * Returns the document source of this element.
         * @return the source of this element.
         */
        DocumentSource getDocumentSource();

    }
}
