/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 * 
 *      http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.commons.lang;

import java.io.IOException;
import java.io.StringWriter;
import java.io.Writer;
import java.util.HashMap;
import java.util.Map;
import java.util.TreeMap;

/**
 *  <p>
 *  Provides HTML and XML entity utilities.
 *  </p>
 *
 *  @see <a href="http://hotwired.lycos.com/webmonkey/reference/special_characters/">ISO Entities</a>
 *  @see <a href="http://www.w3.org/TR/REC-html32#latin1">HTML 3.2 Character Entities for ISO Latin-1</a>
 *  @see <a href="http://www.w3.org/TR/REC-html40/sgml/entities.html">HTML 4.0 Character entity references</a>
 *  @see <a href="http://www.w3.org/TR/html401/charset.html#h-5.3">HTML 4.01 Character References</a>
 *  @see <a href="http://www.w3.org/TR/html401/charset.html#code-position">HTML 4.01 Code positions</a>
 *
 *  @author <a href="mailto:alex@purpletech.com">Alexander Day Chaffee</a>
 *  @author <a href="mailto:ggregory@seagullsw.com">Gary Gregory</a>
 *  @since 2.0
 *  @version $Id: Entities.java 1057037 2011-01-09 21:35:32Z niallp $
 *
 * @deprecated Commons Lang 2 is in maintenance mode. Commons Lang 3 should be used instead.
 */
@Deprecated(since = "2021-04-30")
class Entities {

    private static final String[][] BASIC_ARRAY = { // " - double-quote
    { "quot", "34" }, // & - ampersand
    { "amp", "38" }, // < - less-than
    { "lt", "60" }, // > - greater-than
    { "gt", "62" } };

    private static final String[][] APOS_ARRAY = { // XML apostrophe
    { "apos", "39" } };

    // package scoped for testing
    static final String[][] ISO8859_1_ARRAY = { // non-breaking space
    { "nbsp", "160" }, // inverted exclamation mark
    { "iexcl", "161" }, // cent sign
    { "cent", "162" }, // pound sign
    { "pound", "163" }, // currency sign
    { "curren", "164" }, // yen sign = yuan sign
    { "yen", "165" }, // broken bar = broken vertical bar
    { "brvbar", "166" }, // section sign
    { "sect", "167" }, // diaeresis = spacing diaeresis
    { "uml", "168" }, // © - copyright sign
    { "copy", "169" }, // feminine ordinal indicator
    { "ordf", "170" }, // left-pointing double angle quotation mark = left pointing guillemet
    { "laquo", "171" }, // not sign
    { "not", "172" }, // soft hyphen = discretionary hyphen
    { "shy", "173" }, // ® - registered trademark sign
    { "reg", "174" }, // macron = spacing macron = overline = APL overbar
    { "macr", "175" }, // degree sign
    { "deg", "176" }, // plus-minus sign = plus-or-minus sign
    { "plusmn", "177" }, // superscript two = superscript digit two = squared
    { "sup2", "178" }, // superscript three = superscript digit three = cubed
    { "sup3", "179" }, // acute accent = spacing acute
    { "acute", "180" }, // micro sign
    { "micro", "181" }, // pilcrow sign = paragraph sign
    { "para", "182" }, // middle dot = Georgian comma = Greek middle dot
    { "middot", "183" }, // cedilla = spacing cedilla
    { "cedil", "184" }, // superscript one = superscript digit one
    { "sup1", "185" }, // masculine ordinal indicator
    { "ordm", "186" }, // right-pointing double angle quotation mark = right pointing guillemet
    { "raquo", "187" }, // vulgar fraction one quarter = fraction one quarter
    { "frac14", "188" }, // vulgar fraction one half = fraction one half
    { "frac12", "189" }, // vulgar fraction three quarters = fraction three quarters
    { "frac34", "190" }, // inverted question mark = turned question mark
    { "iquest", "191" }, // À - uppercase A, grave accent
    { "Agrave", "192" }, // Á - uppercase A, acute accent
    { "Aacute", "193" }, // Â - uppercase A, circumflex accent
    { "Acirc", "194" }, // Ã - uppercase A, tilde
    { "Atilde", "195" }, // Ä - uppercase A, umlaut
    { "Auml", "196" }, // Å - uppercase A, ring
    { "Aring", "197" }, // Æ - uppercase AE
    { "AElig", "198" }, // Ç - uppercase C, cedilla
    { "Ccedil", "199" }, // È - uppercase E, grave accent
    { "Egrave", "200" }, // É - uppercase E, acute accent
    { "Eacute", "201" }, // Ê - uppercase E, circumflex accent
    { "Ecirc", "202" }, // Ë - uppercase E, umlaut
    { "Euml", "203" }, // Ì - uppercase I, grave accent
    { "Igrave", "204" }, // Í - uppercase I, acute accent
    { "Iacute", "205" }, // Î - uppercase I, circumflex accent
    { "Icirc", "206" }, // Ï - uppercase I, umlaut
    { "Iuml", "207" }, // Ð - uppercase Eth, Icelandic
    { "ETH", "208" }, // Ñ - uppercase N, tilde
    { "Ntilde", "209" }, // Ò - uppercase O, grave accent
    { "Ograve", "210" }, // Ó - uppercase O, acute accent
    { "Oacute", "211" }, // Ô - uppercase O, circumflex accent
    { "Ocirc", "212" }, // Õ - uppercase O, tilde
    { "Otilde", "213" }, // Ö - uppercase O, umlaut
    { "Ouml", "214" }, // multiplication sign
    { "times", "215" }, // Ø - uppercase O, slash
    { "Oslash", "216" }, // Ù - uppercase U, grave accent
    { "Ugrave", "217" }, // Ú - uppercase U, acute accent
    { "Uacute", "218" }, // Û - uppercase U, circumflex accent
    { "Ucirc", "219" }, // Ü - uppercase U, umlaut
    { "Uuml", "220" }, // Ý - uppercase Y, acute accent
    { "Yacute", "221" }, // Þ - uppercase THORN, Icelandic
    { "THORN", "222" }, // ß - lowercase sharps, German
    { "szlig", "223" }, // à - lowercase a, grave accent
    { "agrave", "224" }, // á - lowercase a, acute accent
    { "aacute", "225" }, // â - lowercase a, circumflex accent
    { "acirc", "226" }, // ã - lowercase a, tilde
    { "atilde", "227" }, // ä - lowercase a, umlaut
    { "auml", "228" }, // å - lowercase a, ring
    { "aring", "229" }, // æ - lowercase ae
    { "aelig", "230" }, // ç - lowercase c, cedilla
    { "ccedil", "231" }, // è - lowercase e, grave accent
    { "egrave", "232" }, // é - lowercase e, acute accent
    { "eacute", "233" }, // ê - lowercase e, circumflex accent
    { "ecirc", "234" }, // ë - lowercase e, umlaut
    { "euml", "235" }, // ì - lowercase i, grave accent
    { "igrave", "236" }, // í - lowercase i, acute accent
    { "iacute", "237" }, // î - lowercase i, circumflex accent
    { "icirc", "238" }, // ï - lowercase i, umlaut
    { "iuml", "239" }, // ð - lowercase eth, Icelandic
    { "eth", "240" }, // ñ - lowercase n, tilde
    { "ntilde", "241" }, // ò - lowercase o, grave accent
    { "ograve", "242" }, // ó - lowercase o, acute accent
    { "oacute", "243" }, // ô - lowercase o, circumflex accent
    { "ocirc", "244" }, // õ - lowercase o, tilde
    { "otilde", "245" }, // ö - lowercase o, umlaut
    { "ouml", "246" }, // division sign
    { "divide", "247" }, // ø - lowercase o, slash
    { "oslash", "248" }, // ù - lowercase u, grave accent
    { "ugrave", "249" }, // ú - lowercase u, acute accent
    { "uacute", "250" }, // û - lowercase u, circumflex accent
    { "ucirc", "251" }, // ü - lowercase u, umlaut
    { "uuml", "252" }, // ý - lowercase y, acute accent
    { "yacute", "253" }, // þ - lowercase thorn, Icelandic
    { "thorn", "254" }, // ÿ - lowercase y, umlaut
    { "yuml", "255" } };

    // http://www.w3.org/TR/REC-html40/sgml/entities.html
    // package scoped for testing
    static final String[][] HTML40_ARRAY = { // <!-- Latin Extended-B -->
    // latin small f with hook = function= florin, U+0192 ISOtech -->
    { "fnof", "402" }, // <!-- Greek -->
    // greek capital letter alpha, U+0391 -->
    { "Alpha", "913" }, // greek capital letter beta, U+0392 -->
    { "Beta", "914" }, // greek capital letter gamma,U+0393 ISOgrk3 -->
    { "Gamma", "915" }, // greek capital letter delta,U+0394 ISOgrk3 -->
    { "Delta", "916" }, // greek capital letter epsilon, U+0395 -->
    { "Epsilon", "917" }, // greek capital letter zeta, U+0396 -->
    { "Zeta", "918" }, // greek capital letter eta, U+0397 -->
    { "Eta", "919" }, // greek capital letter theta,U+0398 ISOgrk3 -->
    { "Theta", "920" }, // greek capital letter iota, U+0399 -->
    { "Iota", "921" }, // greek capital letter kappa, U+039A -->
    { "Kappa", "922" }, // greek capital letter lambda,U+039B ISOgrk3 -->
    { "Lambda", "923" }, // greek capital letter mu, U+039C -->
    { "Mu", "924" }, // greek capital letter nu, U+039D -->
    { "Nu", "925" }, // greek capital letter xi, U+039E ISOgrk3 -->
    { "Xi", "926" }, // greek capital letter omicron, U+039F -->
    { "Omicron", "927" }, // greek capital letter pi, U+03A0 ISOgrk3 -->
    { "Pi", "928" }, // greek capital letter rho, U+03A1 -->
    { "Rho", "929" }, // <!-- there is no Sigmaf, and no U+03A2 character either -->
    // greek capital letter sigma,U+03A3 ISOgrk3 -->
    { "Sigma", "931" }, // greek capital letter tau, U+03A4 -->
    { "Tau", "932" }, // greek capital letter upsilon,U+03A5 ISOgrk3 -->
    { "Upsilon", "933" }, // greek capital letter phi,U+03A6 ISOgrk3 -->
    { "Phi", "934" }, // greek capital letter chi, U+03A7 -->
    { "Chi", "935" }, // greek capital letter psi,U+03A8 ISOgrk3 -->
    { "Psi", "936" }, // greek capital letter omega,U+03A9 ISOgrk3 -->
    { "Omega", "937" }, // greek small letter alpha,U+03B1 ISOgrk3 -->
    { "alpha", "945" }, // greek small letter beta, U+03B2 ISOgrk3 -->
    { "beta", "946" }, // greek small letter gamma,U+03B3 ISOgrk3 -->
    { "gamma", "947" }, // greek small letter delta,U+03B4 ISOgrk3 -->
    { "delta", "948" }, // greek small letter epsilon,U+03B5 ISOgrk3 -->
    { "epsilon", "949" }, // greek small letter zeta, U+03B6 ISOgrk3 -->
    { "zeta", "950" }, // greek small letter eta, U+03B7 ISOgrk3 -->
    { "eta", "951" }, // greek small letter theta,U+03B8 ISOgrk3 -->
    { "theta", "952" }, // greek small letter iota, U+03B9 ISOgrk3 -->
    { "iota", "953" }, // greek small letter kappa,U+03BA ISOgrk3 -->
    { "kappa", "954" }, // greek small letter lambda,U+03BB ISOgrk3 -->
    { "lambda", "955" }, // greek small letter mu, U+03BC ISOgrk3 -->
    { "mu", "956" }, // greek small letter nu, U+03BD ISOgrk3 -->
    { "nu", "957" }, // greek small letter xi, U+03BE ISOgrk3 -->
    { "xi", "958" }, // greek small letter omicron, U+03BF NEW -->
    { "omicron", "959" }, // greek small letter pi, U+03C0 ISOgrk3 -->
    { "pi", "960" }, // greek small letter rho, U+03C1 ISOgrk3 -->
    { "rho", "961" }, // greek small letter final sigma,U+03C2 ISOgrk3 -->
    { "sigmaf", "962" }, // greek small letter sigma,U+03C3 ISOgrk3 -->
    { "sigma", "963" }, // greek small letter tau, U+03C4 ISOgrk3 -->
    { "tau", "964" }, // greek small letter upsilon,U+03C5 ISOgrk3 -->
    { "upsilon", "965" }, // greek small letter phi, U+03C6 ISOgrk3 -->
    { "phi", "966" }, // greek small letter chi, U+03C7 ISOgrk3 -->
    { "chi", "967" }, // greek small letter psi, U+03C8 ISOgrk3 -->
    { "psi", "968" }, // greek small letter omega,U+03C9 ISOgrk3 -->
    { "omega", "969" }, // greek small letter theta symbol,U+03D1 NEW -->
    { "thetasym", "977" }, // greek upsilon with hook symbol,U+03D2 NEW -->
    { "upsih", "978" }, // greek pi symbol, U+03D6 ISOgrk3 -->
    { "piv", "982" }, // <!-- General Punctuation -->
    // bullet = black small circle,U+2022 ISOpub -->
    { "bull", "8226" }, // <!-- bullet is NOT the same as bullet operator, U+2219 -->
    // horizontal ellipsis = three dot leader,U+2026 ISOpub -->
    { "hellip", "8230" }, // prime = minutes = feet, U+2032 ISOtech -->
    { "prime", "8242" }, // double prime = seconds = inches,U+2033 ISOtech -->
    { "Prime", "8243" }, // overline = spacing overscore,U+203E NEW -->
    { "oline", "8254" }, // fraction slash, U+2044 NEW -->
    { "frasl", "8260" }, // <!-- Letterlike Symbols -->
    // script capital P = power set= Weierstrass p, U+2118 ISOamso -->
    { "weierp", "8472" }, // blackletter capital I = imaginary part,U+2111 ISOamso -->
    { "image", "8465" }, // blackletter capital R = real part symbol,U+211C ISOamso -->
    { "real", "8476" }, // trade mark sign, U+2122 ISOnum -->
    { "trade", "8482" }, // alef symbol = first transfinite cardinal,U+2135 NEW -->
    { "alefsym", "8501" }, // <!-- alef symbol is NOT the same as hebrew letter alef,U+05D0 although the
    // same glyph could be used to depict both characters -->
    // <!-- Arrows -->
    // leftwards arrow, U+2190 ISOnum -->
    { "larr", "8592" }, // upwards arrow, U+2191 ISOnum-->
    { "uarr", "8593" }, // rightwards arrow, U+2192 ISOnum -->
    { "rarr", "8594" }, // downwards arrow, U+2193 ISOnum -->
    { "darr", "8595" }, // left right arrow, U+2194 ISOamsa -->
    { "harr", "8596" }, // downwards arrow with corner leftwards= carriage return, U+21B5 NEW -->
    { "crarr", "8629" }, // leftwards double arrow, U+21D0 ISOtech -->
    { "lArr", "8656" }, // <!-- ISO 10646 does not say that lArr is the same as the 'is implied by'
    // arrow but also does not have any other character for that function.
    // So ? lArr canbe used for 'is implied by' as ISOtech suggests -->
    // upwards double arrow, U+21D1 ISOamsa -->
    { "uArr", "8657" }, // rightwards double arrow,U+21D2 ISOtech -->
    { "rArr", "8658" }, // <!-- ISO 10646 does not say this is the 'implies' character but does not
    // have another character with this function so ?rArr can be used for
    // 'implies' as ISOtech suggests -->
    // downwards double arrow, U+21D3 ISOamsa -->
    { "dArr", "8659" }, // left right double arrow,U+21D4 ISOamsa -->
    { "hArr", "8660" }, // <!-- Mathematical Operators -->
    // for all, U+2200 ISOtech -->
    { "forall", "8704" }, // partial differential, U+2202 ISOtech -->
    { "part", "8706" }, // there exists, U+2203 ISOtech -->
    { "exist", "8707" }, // empty set = null set = diameter,U+2205 ISOamso -->
    { "empty", "8709" }, // nabla = backward difference,U+2207 ISOtech -->
    { "nabla", "8711" }, // element of, U+2208 ISOtech -->
    { "isin", "8712" }, // not an element of, U+2209 ISOtech -->
    { "notin", "8713" }, // contains as member, U+220B ISOtech -->
    { "ni", "8715" }, // <!-- should there be a more memorable name than 'ni'? -->
    // n-ary product = product sign,U+220F ISOamsb -->
    { "prod", "8719" }, // <!-- prod is NOT the same character as U+03A0 'greek capital letter pi'
    // though the same glyph might be used for both -->
    // n-ary summation, U+2211 ISOamsb -->
    { "sum", "8721" }, // <!-- sum is NOT the same character as U+03A3 'greek capital letter sigma'
    // though the same glyph might be used for both -->
    // minus sign, U+2212 ISOtech -->
    { "minus", "8722" }, // asterisk operator, U+2217 ISOtech -->
    { "lowast", "8727" }, // square root = radical sign,U+221A ISOtech -->
    { "radic", "8730" }, // proportional to, U+221D ISOtech -->
    { "prop", "8733" }, // infinity, U+221E ISOtech -->
    { "infin", "8734" }, // angle, U+2220 ISOamso -->
    { "ang", "8736" }, // logical and = wedge, U+2227 ISOtech -->
    { "and", "8743" }, // logical or = vee, U+2228 ISOtech -->
    { "or", "8744" }, // intersection = cap, U+2229 ISOtech -->
    { "cap", "8745" }, // union = cup, U+222A ISOtech -->
    { "cup", "8746" }, // integral, U+222B ISOtech -->
    { "int", "8747" }, // therefore, U+2234 ISOtech -->
    { "there4", "8756" }, // tilde operator = varies with = similar to,U+223C ISOtech -->
    { "sim", "8764" }, // <!-- tilde operator is NOT the same character as the tilde, U+007E,although
    // the same glyph might be used to represent both -->
    // approximately equal to, U+2245 ISOtech -->
    { "cong", "8773" }, // almost equal to = asymptotic to,U+2248 ISOamsr -->
    { "asymp", "8776" }, // not equal to, U+2260 ISOtech -->
    { "ne", "8800" }, // identical to, U+2261 ISOtech -->
    { "equiv", "8801" }, // less-than or equal to, U+2264 ISOtech -->
    { "le", "8804" }, // greater-than or equal to,U+2265 ISOtech -->
    { "ge", "8805" }, // subset of, U+2282 ISOtech -->
    { "sub", "8834" }, // superset of, U+2283 ISOtech -->
    { "sup", "8835" }, // <!-- note that nsup, 'not a superset of, U+2283' is not covered by the
    // Symbol font encoding and is not included. Should it be, for symmetry?
    // It is in ISOamsn --> <!ENTITY nsub", "8836"},
    // not a subset of, U+2284 ISOamsn -->
    // subset of or equal to, U+2286 ISOtech -->
    { "sube", "8838" }, // superset of or equal to,U+2287 ISOtech -->
    { "supe", "8839" }, // circled plus = direct sum,U+2295 ISOamsb -->
    { "oplus", "8853" }, // circled times = vector product,U+2297 ISOamsb -->
    { "otimes", "8855" }, // up tack = orthogonal to = perpendicular,U+22A5 ISOtech -->
    { "perp", "8869" }, // dot operator, U+22C5 ISOamsb -->
    { "sdot", "8901" }, // <!-- dot operator is NOT the same character as U+00B7 middle dot -->
    // <!-- Miscellaneous Technical -->
    // left ceiling = apl upstile,U+2308 ISOamsc -->
    { "lceil", "8968" }, // right ceiling, U+2309 ISOamsc -->
    { "rceil", "8969" }, // left floor = apl downstile,U+230A ISOamsc -->
    { "lfloor", "8970" }, // right floor, U+230B ISOamsc -->
    { "rfloor", "8971" }, // left-pointing angle bracket = bra,U+2329 ISOtech -->
    { "lang", "9001" }, // <!-- lang is NOT the same character as U+003C 'less than' or U+2039 'single left-pointing angle quotation
    // mark' -->
    // right-pointing angle bracket = ket,U+232A ISOtech -->
    { "rang", "9002" }, // <!-- rang is NOT the same character as U+003E 'greater than' or U+203A
    // 'single right-pointing angle quotation mark' -->
    // <!-- Geometric Shapes -->
    // lozenge, U+25CA ISOpub -->
    { "loz", "9674" }, // <!-- Miscellaneous Symbols -->
    // black spade suit, U+2660 ISOpub -->
    { "spades", "9824" }, // <!-- black here seems to mean filled as opposed to hollow -->
    // black club suit = shamrock,U+2663 ISOpub -->
    { "clubs", "9827" }, // black heart suit = valentine,U+2665 ISOpub -->
    { "hearts", "9829" }, // black diamond suit, U+2666 ISOpub -->
    { "diams", "9830" }, // <!-- Latin Extended-A -->
    // -- latin capital ligature OE,U+0152 ISOlat2 -->
    { "OElig", "338" }, // -- latin small ligature oe, U+0153 ISOlat2 -->
    { "oelig", "339" }, // <!-- ligature is a misnomer, this is a separate character in some languages -->
    // -- latin capital letter S with caron,U+0160 ISOlat2 -->
    { "Scaron", "352" }, // -- latin small letter s with caron,U+0161 ISOlat2 -->
    { "scaron", "353" }, // -- latin capital letter Y with diaeresis,U+0178 ISOlat2 -->
    { "Yuml", "376" }, // <!-- Spacing Modifier Letters -->
    // -- modifier letter circumflex accent,U+02C6 ISOpub -->
    { "circ", "710" }, // small tilde, U+02DC ISOdia -->
    { "tilde", "732" }, // <!-- General Punctuation -->
    // en space, U+2002 ISOpub -->
    { "ensp", "8194" }, // em space, U+2003 ISOpub -->
    { "emsp", "8195" }, // thin space, U+2009 ISOpub -->
    { "thinsp", "8201" }, // zero width non-joiner,U+200C NEW RFC 2070 -->
    { "zwnj", "8204" }, // zero width joiner, U+200D NEW RFC 2070 -->
    { "zwj", "8205" }, // left-to-right mark, U+200E NEW RFC 2070 -->
    { "lrm", "8206" }, // right-to-left mark, U+200F NEW RFC 2070 -->
    { "rlm", "8207" }, // en dash, U+2013 ISOpub -->
    { "ndash", "8211" }, // em dash, U+2014 ISOpub -->
    { "mdash", "8212" }, // left single quotation mark,U+2018 ISOnum -->
    { "lsquo", "8216" }, // right single quotation mark,U+2019 ISOnum -->
    { "rsquo", "8217" }, // single low-9 quotation mark, U+201A NEW -->
    { "sbquo", "8218" }, // left double quotation mark,U+201C ISOnum -->
    { "ldquo", "8220" }, // right double quotation mark,U+201D ISOnum -->
    { "rdquo", "8221" }, // double low-9 quotation mark, U+201E NEW -->
    { "bdquo", "8222" }, // dagger, U+2020 ISOpub -->
    { "dagger", "8224" }, // double dagger, U+2021 ISOpub -->
    { "Dagger", "8225" }, // per mille sign, U+2030 ISOtech -->
    { "permil", "8240" }, // single left-pointing angle quotation mark,U+2039 ISO proposed -->
    { "lsaquo", "8249" }, // <!-- lsaquo is proposed but not yet ISO standardized -->
    // single right-pointing angle quotation mark,U+203A ISO proposed -->
    { "rsaquo", "8250" }, // <!-- rsaquo is proposed but not yet ISO standardized -->
    // -- euro sign, U+20AC NEW -->
    { "euro", "8364" } };

    /**
     * <p>
     * The set of entities supported by standard XML.
     * </p>
     */
    public static final Entities XML;

    /**
     * <p>
     * The set of entities supported by HTML 3.2.
     * </p>
     */
    public static final Entities HTML32;

    /**
     * <p>
     * The set of entities supported by HTML 4.0.
     * </p>
     */
    public static final Entities HTML40;

    static {
        Entities xml = new Entities();
        xml.addEntities(BASIC_ARRAY);
        xml.addEntities(APOS_ARRAY);
        XML = xml;
    }

    static {
        Entities html32 = new Entities();
        html32.addEntities(BASIC_ARRAY);
        html32.addEntities(ISO8859_1_ARRAY);
        HTML32 = html32;
    }

    static {
        Entities html40 = new Entities();
        fillWithHtml40Entities(html40);
        HTML40 = html40;
    }

    /**
     * <p>
     * Fills the specified entities instance with HTML 40 entities.
     * </p>
     *
     * @param entities
     *            the instance to be filled.
     */
    static void fillWithHtml40Entities(Entities entities) {
        entities.addEntities(BASIC_ARRAY);
        entities.addEntities(ISO8859_1_ARRAY);
        entities.addEntities(HTML40_ARRAY);
    }

    // @deprecated Commons Lang 2 is in maintenance mode. Commons Lang 3 should be used instead.
    @Deprecated(since = "2021-04-30")
    static interface EntityMap {

        /**
         * <p>
         * Add an entry to this entity map.
         * </p>
         *
         * @param name
         *            the entity name
         * @param value
         *            the entity value
         */
        void add(String name, int value);

        /**
         * <p>
         * Returns the name of the entity identified by the specified value.
         * </p>
         *
         * @param value
         *            the value to locate
         * @return entity name associated with the specified value
         */
        String name(int value);

        /**
         * <p>
         * Returns the value of the entity identified by the specified name.
         * </p>
         *
         * @param name
         *            the name to locate
         * @return entity value associated with the specified name
         */
        int value(String name);
    }

    // @deprecated Commons Lang 2 is in maintenance mode. Commons Lang 3 should be used instead.
    @Deprecated(since = "2021-04-30")
    static class PrimitiveEntityMap implements EntityMap {

        private final Map mapNameToValue = new HashMap();

        private final IntHashMap mapValueToName = new IntHashMap();

        /**
         * {@inheritDoc}
         */
        // TODO not thread-safe as there is a window between changing the two maps
        public void add(String name, int value) {
            mapNameToValue.put(name, new Integer(value));
            mapValueToName.put(value, name);
        }

        /**
         * {@inheritDoc}
         */
        public String name(int value) {
            return (String) mapValueToName.get(value);
        }

        /**
         * {@inheritDoc}
         */
        public int value(String name) {
            Object value = mapNameToValue.get(name);
            if (value == null) {
                return -1;
            }
            return ((Integer) value).intValue();
        }
    }

    // @deprecated Commons Lang 2 is in maintenance mode. Commons Lang 3 should be used instead.
    @Deprecated(since = "2021-04-30")
    static abstract class MapIntMap implements Entities.EntityMap {

        protected final Map mapNameToValue;

        protected final Map mapValueToName;

        /**
         * Construct a new instance with specified maps.
         *
         * @param nameToValue name to value map
         * @param valueToName value to namee map
         */
        MapIntMap(Map nameToValue, Map valueToName) {
            mapNameToValue = nameToValue;
            mapValueToName = valueToName;
        }

        /**
         * {@inheritDoc}
         */
        public void add(String name, int value) {
            mapNameToValue.put(name, new Integer(value));
            mapValueToName.put(new Integer(value), name);
        }

        /**
         * {@inheritDoc}
         */
        public String name(int value) {
            return (String) mapValueToName.get(new Integer(value));
        }

        /**
         * {@inheritDoc}
         */
        public int value(String name) {
            Object value = mapNameToValue.get(name);
            if (value == null) {
                return -1;
            }
            return ((Integer) value).intValue();
        }
    }

    // @deprecated Commons Lang 2 is in maintenance mode. Commons Lang 3 should be used instead.
    @Deprecated(since = "2021-04-30")
    static class HashEntityMap extends MapIntMap {

        /**
         * Constructs a new instance of <code>HashEntityMap</code>.
         */
        public HashEntityMap() {
            super(new HashMap(), new HashMap());
        }
    }

    // @deprecated Commons Lang 2 is in maintenance mode. Commons Lang 3 should be used instead.
    @Deprecated(since = "2021-04-30")
    static class TreeEntityMap extends MapIntMap {

        /**
         * Constructs a new instance of <code>TreeEntityMap</code>.
         */
        public TreeEntityMap() {
            super(new TreeMap(), new TreeMap());
        }
    }

    // @deprecated Commons Lang 2 is in maintenance mode. Commons Lang 3 should be used instead.
    @Deprecated(since = "2021-04-30")
    static class LookupEntityMap extends PrimitiveEntityMap {

        // TODO this class is not thread-safe
        private String[] lookupTable;

        private static final int LOOKUP_TABLE_SIZE = 256;

        /**
         * {@inheritDoc}
         */
        public String name(int value) {
            if (value < LOOKUP_TABLE_SIZE) {
                return lookupTable()[value];
            }
            return super.name(value);
        }

        /**
         * <p>
         * Returns the lookup table for this entity map. The lookup table is created if it has not been previously.
         * </p>
         *
         * @return the lookup table
         */
        private String[] lookupTable() {
            if (lookupTable == null) {
                createLookupTable();
            }
            return lookupTable;
        }

        /**
         * <p>
         * Creates an entity lookup table of LOOKUP_TABLE_SIZE elements, initialized with entity names.
         * </p>
         */
        private void createLookupTable() {
            lookupTable = new String[LOOKUP_TABLE_SIZE];
            for (int i = 0; i < LOOKUP_TABLE_SIZE; ++i) {
                lookupTable[i] = super.name(i);
            }
        }
    }

    // @deprecated Commons Lang 2 is in maintenance mode. Commons Lang 3 should be used instead.
    @Deprecated(since = "2021-04-30")
    static class ArrayEntityMap implements EntityMap {

        // TODO this class is not thread-safe
        protected final int growBy;

        protected int size = 0;

        protected String[] names;

        protected int[] values;

        /**
         * Constructs a new instance of <code>ArrayEntityMap</code>.
         */
        public ArrayEntityMap() {
            this.growBy = 100;
            names = new String[growBy];
            values = new int[growBy];
        }

        /**
         * Constructs a new instance of <code>ArrayEntityMap</code> specifying the size by which the array should
         * grow.
         *
         * @param growBy
         *            array will be initialized to and will grow by this amount
         */
        public ArrayEntityMap(int growBy) {
            this.growBy = growBy;
            names = new String[growBy];
            values = new int[growBy];
        }

        /**
         * {@inheritDoc}
         */
        public void add(String name, int value) {
            ensureCapacity(size + 1);
            names[size] = name;
            values[size] = value;
            size++;
        }

        /**
         * Verifies the capacity of the entity array, adjusting the size if necessary.
         *
         * @param capacity
         *            size the array should be
         */
        protected void ensureCapacity(int capacity) {
            if (capacity > names.length) {
                int newSize = Math.max(capacity, size + growBy);
                String[] newNames = new String[newSize];
                System.arraycopy(names, 0, newNames, 0, size);
                names = newNames;
                int[] newValues = new int[newSize];
                System.arraycopy(values, 0, newValues, 0, size);
                values = newValues;
            }
        }

        /**
         * {@inheritDoc}
         */
        public String name(int value) {
            for (int i = 0; i < size; ++i) {
                if (values[i] == value) {
                    return names[i];
                }
            }
            return null;
        }

        /**
         * {@inheritDoc}
         */
        public int value(String name) {
            for (int i = 0; i < size; ++i) {
                if (names[i].equals(name)) {
                    return values[i];
                }
            }
            return -1;
        }
    }

    // @deprecated Commons Lang 2 is in maintenance mode. Commons Lang 3 should be used instead.
    @Deprecated(since = "2021-04-30")
    static class BinaryEntityMap extends ArrayEntityMap {

        // TODO - not thread-safe, because parent is not. Also references size.
        /**
         * Constructs a new instance of <code>BinaryEntityMap</code>.
         */
        public BinaryEntityMap() {
            super();
        }

        /**
         * Constructs a new instance of <code>ArrayEntityMap</code> specifying the size by which the underlying array
         * should grow.
         *
         * @param growBy
         *            array will be initialized to and will grow by this amount
         */
        public BinaryEntityMap(int growBy) {
            super(growBy);
        }

        /**
         * Performs a binary search of the entity array for the specified key. This method is based on code in
         * {@link java.util.Arrays}.
         *
         * @param key
         *            the key to be found
         * @return the index of the entity array matching the specified key
         */
        private int binarySearch(int key) {
            int low = 0;
            int high = size - 1;
            while (low <= high) {
                int mid = (low + high) >>> 1;
                int midVal = values[mid];
                if (midVal < key) {
                    low = mid + 1;
                } else if (midVal > key) {
                    high = mid - 1;
                } else {
                    // key found
                    return mid;
                }
            }
            // key not found.
            return -(low + 1);
        }

        /**
         * {@inheritDoc}
         */
        public void add(String name, int value) {
            ensureCapacity(size + 1);
            int insertAt = binarySearch(value);
            if (insertAt > 0) {
                // note: this means you can't insert the same value twice
                return;
            }
            // binarySearch returns it negative and off-by-one
            insertAt = -(insertAt + 1);
            System.arraycopy(values, insertAt, values, insertAt + 1, size - insertAt);
            values[insertAt] = value;
            System.arraycopy(names, insertAt, names, insertAt + 1, size - insertAt);
            names[insertAt] = name;
            size++;
        }

        /**
         * {@inheritDoc}
         */
        public String name(int value) {
            int index = binarySearch(value);
            if (index < 0) {
                return null;
            }
            return names[index];
        }
    }

    private final EntityMap map;

    /**
     * Default constructor.
     */
    public Entities() {
        map = new Entities.LookupEntityMap();
    }

    /**
     * package scoped constructor for testing.
     *
     * @param emap entity map.
     */
    Entities(EntityMap emap) {
        map = emap;
    }

    /**
     * <p>
     * Adds entities to this entity.
     * </p>
     *
     * @param entityArray
     *            array of entities to be added
     */
    public void addEntities(String[][] entityArray) {
        for (int i = 0; i < entityArray.length; ++i) {
            addEntity(entityArray[i][0], Integer.parseInt(entityArray[i][1]));
        }
    }

    /**
     * <p>
     * Add an entity to this entity.
     * </p>
     *
     * @param name
     *            name of the entity
     * @param value
     *            vale of the entity
     */
    public void addEntity(String name, int value) {
        map.add(name, value);
    }

    /**
     * <p>
     * Returns the name of the entity identified by the specified value.
     * </p>
     *
     * @param value
     *            the value to locate
     * @return entity name associated with the specified value
     */
    public String entityName(int value) {
        return map.name(value);
    }

    /**
     * <p>
     * Returns the value of the entity identified by the specified name.
     * </p>
     *
     * @param name
     *            the name to locate
     * @return entity value associated with the specified name
     */
    public int entityValue(String name) {
        return map.value(name);
    }

    /**
     * <p>
     * Escapes the characters in a <code>String</code>.
     * </p>
     *
     * <p>
     * For example, if you have called addEntity(&quot;foo&quot;, 0xA1), escape(&quot;\u00A1&quot;) will return
     * &quot;&amp;foo;&quot;
     * </p>
     *
     * @param str
     *            The <code>String</code> to escape.
     * @return A new escaped <code>String</code>.
     */
    public String escape(String str) {
        StringWriter stringWriter = createStringWriter(str);
        try {
            this.escape(stringWriter, str);
        } catch (IOException e) {
            // This should never happen because ALL the StringWriter methods called by #escape(Writer, String) do not
            // throw IOExceptions.
            throw new UnhandledException(e);
        }
        return stringWriter.toString();
    }

    /**
     * <p>
     * Escapes the characters in the <code>String</code> passed and writes the result to the <code>Writer</code>
     * passed.
     * </p>
     *
     * @param writer
     *            The <code>Writer</code> to write the results of the escaping to. Assumed to be a non-null value.
     * @param str
     *            The <code>String</code> to escape. Assumed to be a non-null value.
     * @throws IOException
     *             when <code>Writer</code> passed throws the exception from calls to the {@link Writer#write(int)}
     *             methods.
     *
     * @see #escape(String)
     * @see Writer
     */
    public void escape(Writer writer, String str) throws IOException {
        int len = str.length();
        for (int i = 0; i < len; i++) {
            char c = str.charAt(i);
            String entityName = this.entityName(c);
            if (entityName == null) {
                if (c > 0x7F) {
                    writer.write("&#");
                    writer.write(Integer.toString(c, 10));
                    writer.write(';');
                } else {
                    writer.write(c);
                }
            } else {
                writer.write('&');
                writer.write(entityName);
                writer.write(';');
            }
        }
    }

    /**
     * <p>
     * Unescapes the entities in a <code>String</code>.
     * </p>
     *
     * <p>
     * For example, if you have called addEntity(&quot;foo&quot;, 0xA1), unescape(&quot;&amp;foo;&quot;) will return
     * &quot;\u00A1&quot;
     * </p>
     *
     * @param str
     *            The <code>String</code> to escape.
     * @return A new escaped <code>String</code>.
     */
    public String unescape(String str) {
        int firstAmp = str.indexOf('&');
        if (firstAmp < 0) {
            return str;
        } else {
            StringWriter stringWriter = createStringWriter(str);
            try {
                this.doUnescape(stringWriter, str, firstAmp);
            } catch (IOException e) {
                // This should never happen because ALL the StringWriter methods called by #escape(Writer, String)
                // do not throw IOExceptions.
                throw new UnhandledException(e);
            }
            return stringWriter.toString();
        }
    }

    /**
     * Make the StringWriter 10% larger than the source String to avoid growing the writer
     *
     * @param str The source string
     * @return A newly created StringWriter
     */
    private StringWriter createStringWriter(String str) {
        return new StringWriter((int) (str.length() + (str.length() * 0.1)));
    }

    /**
     * <p>
     * Unescapes the escaped entities in the <code>String</code> passed and writes the result to the
     * <code>Writer</code> passed.
     * </p>
     *
     * @param writer
     *            The <code>Writer</code> to write the results to; assumed to be non-null.
     * @param str
     *            The source <code>String</code> to unescape; assumed to be non-null.
     * @throws IOException
     *             when <code>Writer</code> passed throws the exception from calls to the {@link Writer#write(int)}
     *             methods.
     *
     * @see #escape(String)
     * @see Writer
     */
    public void unescape(Writer writer, String str) throws IOException {
        int firstAmp = str.indexOf('&');
        if (firstAmp < 0) {
            writer.write(str);
            return;
        } else {
            doUnescape(writer, str, firstAmp);
        }
    }

    /**
     * Underlying unescape method that allows the optimisation of not starting from the 0 index again.
     *
     * @param writer
     *            The <code>Writer</code> to write the results to; assumed to be non-null.
     * @param str
     *            The source <code>String</code> to unescape; assumed to be non-null.
     * @param firstAmp
     *            The <code>int</code> index of the first ampersand in the source String.
     * @throws IOException
     *             when <code>Writer</code> passed throws the exception from calls to the {@link Writer#write(int)}
     *             methods.
     */
    private void doUnescape(Writer writer, String str, int firstAmp) throws IOException {
        writer.write(str, 0, firstAmp);
        int len = str.length();
        for (int i = firstAmp; i < len; i++) {
            char c = str.charAt(i);
            if (c == '&') {
                int nextIdx = i + 1;
                int semiColonIdx = str.indexOf(';', nextIdx);
                if (semiColonIdx == -1) {
                    writer.write(c);
                    continue;
                }
                int amphersandIdx = str.indexOf('&', i + 1);
                if (amphersandIdx != -1 && amphersandIdx < semiColonIdx) {
                    // Then the text looks like &...&...;
                    writer.write(c);
                    continue;
                }
                String entityContent = str.substring(nextIdx, semiColonIdx);
                int entityValue = -1;
                int entityContentLen = entityContent.length();
                if (entityContentLen > 0) {
                    if (entityContent.charAt(0) == '#') {
                        // escaped value content is an integer (decimal or
                        // hexidecimal)
                        if (entityContentLen > 1) {
                            char isHexChar = entityContent.charAt(1);
                            try {
                                switch(isHexChar) {
                                    case 'X':
                                    case 'x':
                                        {
                                            entityValue = Integer.parseInt(entityContent.substring(2), 16);
                                            break;
                                        }
                                    default:
                                        {
                                            entityValue = Integer.parseInt(entityContent.substring(1), 10);
                                        }
                                }
                                if (entityValue > 0xFFFF) {
                                    entityValue = -1;
                                }
                            } catch (NumberFormatException e) {
                                entityValue = -1;
                            }
                        }
                    } else {
                        // escaped value content is an entity name
                        entityValue = this.entityValue(entityContent);
                    }
                }
                if (entityValue == -1) {
                    writer.write('&');
                    writer.write(entityContent);
                    writer.write(';');
                } else {
                    writer.write(entityValue);
                }
                // move index up to the semi-colon
                i = semiColonIdx;
            } else {
                writer.write(c);
            }
        }
    }
}
