
/* ====================================================================
   Licensed to the Apache Software Foundation (ASF) under one or more
   contributor license agreements.  See the NOTICE file distributed with
   this work for additional information regarding copyright ownership.
   The ASF licenses this file to You under the Apache License, Version 2.0
   (the "License"); you may not use this file except in compliance with
   the License.  You may obtain a copy of the License at

       http://www.apache.org/licenses/LICENSE-2.0

   Unless required by applicable law or agreed to in writing, software
   distributed under the License is distributed on an "AS IS" BASIS,
   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
   See the License for the specific language governing permissions and
   limitations under the License.
==================================================================== */


package org.apache.poi.poifs.filesystem;

import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.util.Iterator;
import java.util.Set;

import org.apache.poi.hpsf.ClassID;

/**
 * This interface defines methods specific to Directory objects
 * managed by a Filesystem instance.
 */

public interface DirectoryEntry extends Entry, Iterable<Entry> {

    /**
     * get an iterator of the Entry instances contained directly in
     * this instance (in other words, children only; no grandchildren
     * etc.)
     *
     * @return iterator; never null, but hasNext() may return false
     *         immediately (i.e., this DirectoryEntry is empty). All
     *         objects retrieved by next() are guaranteed to be
     *         implementations of Entry.
     */

    Iterator<Entry> getEntries();

    /**
     * get the names of all the Entries contained directly in this
     * instance (in other words, names of children only; no grandchildren
     * etc).
     *
     * @return the names of all the entries that may be retrieved with
     *         getEntry(String), which may be empty (if this
     *         DirectoryEntry is empty)
     */
    Set<String> getEntryNames();

    /**
     * is this DirectoryEntry empty?
     *
     * @return true if this instance contains no Entry instances
     */

    boolean isEmpty();

    /**
     * find out how many Entry instances are contained directly within
     * this DirectoryEntry
     *
     * @return number of immediately (no grandchildren etc.) contained
     *         Entry instances
     */

    int getEntryCount();

    /**
     * Checks if entry with specified name present
     */

    boolean hasEntry( final String name );

    /**
     * get a specified Entry by name
     *
     * @param name the name of the Entry to obtain.
     *
     * @return the specified Entry, if it is directly contained in
     *         this DirectoryEntry
     *
     * @throws FileNotFoundException if no Entry with the specified
     *            name exists in this DirectoryEntry
     */

    Entry getEntry(final String name) throws FileNotFoundException;

    /**
     * create a new DocumentEntry
     *
     * @param name the name of the new DocumentEntry
     * @param stream the InputStream from which to create the new
     *               DocumentEntry
     *
     * @return the new DocumentEntry
     */

    DocumentEntry createDocument(final String name, final InputStream stream)
        throws IOException;

    /**
     * create a new DocumentEntry; the data will be provided later
     *
     * @param name the name of the new DocumentEntry
     * @param size the size of the new DocumentEntry
     * @param writer the writer of the new DocumentEntry
     *
     * @return the new DocumentEntry
     */

    DocumentEntry createDocument(final String name, final int size, final POIFSWriterListener writer)
        throws IOException;

    /**
     * create a new DirectoryEntry
     *
     * @param name the name of the new DirectoryEntry
     *
     * @return the new DirectoryEntry
     */
    DirectoryEntry createDirectory(final String name) throws IOException;

    /**
     * Gets the storage clsid of the directory entry
     *
     * @return storage Class ID
     */
    ClassID getStorageClsid();

    /**
     * Sets the storage clsid for the directory entry
     *
     * @param clsidStorage storage Class ID
     */
    void setStorageClsid(ClassID clsidStorage);
}

