/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * __________________
 *
 *  Copyright 2012 Adobe Systems Incorporated
 *  All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Systems Incorporated and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Systems Incorporated.
 **************************************************************************/
package com.adobe.granite.activitystreams.utils;

import javax.jcr.RepositoryException;
import javax.jcr.Session;

import org.apache.sling.api.resource.Resource;
import org.apache.sling.api.resource.ResourceResolver;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.adobe.granite.activitystreams.ActivityStream;
import com.adobe.granite.activitystreams.ActivityStreamFilter;

/**
 * <code>GenericStreamFilter</code>
 * @since 0.0.10
 */
public class GenericStreamFilter implements ActivityStreamFilter {

    /**
     * default logger
     */
    private static final Logger log = LoggerFactory.getLogger(GenericStreamFilter.class);

    /**
     * Defines the access control mode to include in the filter
     */
    public static enum MODE {
        /**
         * List streams that are at least readable. This is the default.
         */
        R,

        /**
         * List streams that are readable and writable
         */
        RW,

        /**
         * List streams that are readable but not writable
         */
        RO
    }

    private MODE mode = MODE.R;

    /**
     * {@inheritDoc}
     */
    public boolean includes(ActivityStream stream) {
        if (mode == MODE.R) {
            // all streams given here are readable.
            return true;
        }
        // check writable, use session of stream
        Resource r = stream.getContainerResource();
        ResourceResolver resolver = r == null ? null : r.getResourceResolver();
        Session s = resolver == null ? null : resolver.adaptTo(Session.class);
        boolean writable = false;
        if (s != null) {
            try {
                writable = s.hasPermission(stream.getId() + "/__dummy_node__", Session.ACTION_ADD_NODE);
            } catch (RepositoryException e) {
                log.error("Error while evaluating acl on {}", stream.getId(), e);
            }
        }
        if (mode == MODE.RO) {
            return !writable;
        } else /* if (mode == MODE.RW)*/ {
            return writable;
        }
    }

    /**
     * Returns the defined mode.
     * @return the mode.
     */
    public MODE getMode() {
        return mode;
    }

    /**
     * Sets the access control mode for this filter. see {@link GenericStreamFilter.MODE} for details.
     * @param mode the mode
     * @return this filter for chaining.
     */
    public GenericStreamFilter setMode(MODE mode) {
        this.mode = mode;
        return this;
    }
}