package com.adobe.xfa.text;

import com.adobe.xfa.text.markup.MarkupOut;
import com.adobe.xfa.ut.UniCharIterator;

/**
 * @exclude from published api.
 */

class StrText extends StrItem {
	private final StringBuilder msText;
	private final UniCharIterator mIterator;
	private int mnCharIndex;

//----------------------------------------------------------------------
//
//		Standard constuctor: create a stream text item, given a
//		text string.
//
//		Note: this method currently houses the implementation for
//		converting carriage-return/line-feed pairs into new-lines
//		(line-feeds).  This saves a lot of duplication of code
//		elsewhere.	This may not be appropriate in a
//		multi-platform environment.
//
//----------------------------------------------------------------------
	StrText (String sNewText) {
		super (TextItem.CHAR);
		msText = new StringBuilder (sNewText);
		mIterator = new UniCharIterator (msText);
		int charCount = 0;
		int cPrev = '\0';
		for (int c = mIterator.next(); c != '\0'; c = mIterator.next()) {
			if ((cPrev == '\r') && (c == '\n')) {
				mIterator.prev();									// before '\n'
				mIterator.prev();									// before '\r'
				msText.deleteCharAt (mIterator.getIndex());
				mIterator.attach (msText, mIterator.getIndex());	// because iterator caches length
				mIterator.next();									// after '\n' which was counted on '\r'
			} else {
				charCount++;
			}
			cPrev = c;
		}
		setCount (charCount);
		mIterator.setIndex (0);
		mnCharIndex = 0;
	}

	TextAttr markup (MarkupOut oMarkup, int nStart, int nSize, boolean bFlattenFields, TextAttr poPrevAttr) {
		int nStartU16 = findChar (nStart);
		int nAfterU16 = findChar (nStart + nSize);

		if ((nStartU16 == 0) && (nAfterU16 == msText.length())) {
			oMarkup.text (msText.toString());
		} else {
			oMarkup.text (msText.substring (nStartU16, nAfterU16));
		}
		return poPrevAttr;
	}

	int charAt (int nIndex) {
		findChar (nIndex);
		int c = mIterator.next();
		if (c == '\0') {
			assert (false);
		}
		mnCharIndex++;
		return c;
	}

	boolean isEqual (StrItem poCompare) {
		if (! (poCompare instanceof StrText)) {
			return false;
		}
		StrText poText = (StrText) poCompare;
		return msText.equals (poText.msText);
	}

	StrItem[] split (int nIndex) {
		int nU16 = findChar (nIndex);
		StrItem[] pair = new StrItem[2];
		pair[0] = new StrText (msText.substring (0, nU16));
		pair[1] = new StrText (msText.substring (nU16));
		return pair;
	}

	boolean coalesce (StrItem poAfter, int nIndex) {
		StrText poTextItem = (StrText) poAfter;
		msText.append (poTextItem.msText);
		setCount (count() + poTextItem.count());
		mIterator.attach (msText);					// force it to re-get length
		mnCharIndex = 0;
		return false;
	}

	boolean canCoalesce (StrItem poAfter) {
		return true;
	}

	void delete (int nStart, int nSize) {
		int nStartU16 = findChar (nStart);
		int nAfterU16 = findChar (nStart + nSize);
		msText.delete (nStartU16, nAfterU16);
		setCount (count() - nSize);
		mnCharIndex = nStart;
		mIterator.attach (msText, nStartU16);		// force it to re-get length
	}

	StrItem cloneItem (TextGfxSource oGfxSource) {
		return new StrText (msText.toString());
	}

	StrItem clonePart (TextGfxSource oGfxSource, int nStart, int nSize) {
		int nStartU16 = findChar (nStart);
		int nAfterU16 = findChar (nStart + nSize);
		if ((nStartU16 == 0) && (nAfterU16 == msText.length())) {
			return new StrText (msText.toString());
		} else {
			return new StrText (msText.substring (nStartU16, nAfterU16));
		}
	}

// For debugging.
	void debug (int indent) {
		String prefix = Pkg.doIndent (indent+1);
		if (msText.length() == 0) {
			System.out.println (prefix + "Text");
		} else {
			System.out.print (prefix + "Text \"");
			for (int i = 0; i < msText.length(); i++) {
				char c = msText.charAt (i);
				if ((c >= 0x20) && (c < 0x7F)) {
					System.out.print (c);
				} else {
					int ic = ((int) c) & 0xFFFF;
					String s = Integer.toHexString (ic + 0x10000);
					System.out.print ("\\u" + s.substring (1));
				}
			}
			System.out.println ("\"");
		}
	}

	private int findChar (int nIndex) {
		assert (nIndex <= count());

		int nChar;
		int nU16;
		boolean bForward;

		if (nIndex < mnCharIndex) {
			if (nIndex <= (mnCharIndex / 2)) {
				bForward = true;
				nChar = 0;
				nU16 = 0;
			} else {
				bForward = false;
				nChar = mnCharIndex;
				nU16 = mIterator.getIndex();
			}
		} else {
			int nAfter = count() - mnCharIndex;
			if (nIndex <= (mnCharIndex + (nAfter / 2))) {
				bForward = true;
				nChar = mnCharIndex;
				nU16 = mIterator.getIndex();
			} else {
				bForward = false;
				nChar = count();
				nU16 = msText.length();
			}
		}

		mIterator.setIndex (nU16);

		if (bForward) {
			for (; nChar < nIndex; nChar++) {
				mIterator.next();
			}
		} else {
			for (; nChar > nIndex; nChar--) {
				mIterator.prev();
			}
		}

		mnCharIndex = nChar;
		return mIterator.getIndex();
	}
}
