/*~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
 ~ Copyright 2016 Adobe Systems Incorporated
 ~
 ~ Licensed under the Apache License, Version 2.0 (the "License");
 ~ you may not use this file except in compliance with the License.
 ~ You may obtain a copy of the License at
 ~
 ~     http://www.apache.org/licenses/LICENSE-2.0
 ~
 ~ Unless required by applicable law or agreed to in writing, software
 ~ distributed under the License is distributed on an "AS IS" BASIS,
 ~ WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 ~ See the License for the specific language governing permissions and
 ~ limitations under the License.
 ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~*/

package com.day.cq.wcm.foundation.forms;

import org.apache.sling.api.resource.Resource;

/**
 * Interface for retrieving form properties from different form node structures. A form node structure consists
 * of following kinds of nodes:
 * <ol>
 *     <li>
 *         Form node which has form properties like {@link FormsConstants#START_PROPERTY_ACTION_TYPE},
 * {@link FormsConstants#START_PROPERTY_LOAD_PATH}.
 *     </li>
 *     <li>
 *         Form field node which correspond to various input elements which can be present in form.
 *     </li>
 * </ol>
 *
 * Multiple implementations of this interface may be provided. Highest ranked {@link FormStructureHelper} whose
 * {@link #canManage(Resource)} returns true for a given resource, will get used.
 *
 * For getting an instance of this interface check {@link FormStructureHelperFactory}
 */
public interface FormStructureHelper {

    /**
     * Tests whether this {@link FormStructureHelper} can obtain form properties from the given node.
     * @param resource {@link Resource}.
     * @return true if this {@link FormStructureHelper} can obtain form properties, false otherwise.
     */
    boolean canManage(Resource resource);

    /**
     * Gets the resource corresponding to form element i.e. the resource with form properties.
     * @param resource {@link Resource}
     * @return form element {@link Resource} or null if no form resource is found.
     */
    Resource getFormResource(Resource resource);

    /**
     * Gets the resources corresponding to fields of given form resource.
     * @param resource {@link Resource}.
     * @return form field {@link Resource}s.
     */
    Iterable<Resource> getFormElements(final Resource resource);

    /**
     * Updates necessary changes to the form resource. Can be used to set minimal default values for the form (e.g. action type) and to
     * ensure additional mandatory resources (e.g. form start and form end resource)
     *
     * @param resource {@link Resource} any resource.
     * @return updated form resource or null if it wasn't a form related resource or nothing has to be adapted.
     */
    Resource updateFormStructure(final Resource resource);

}
