/*
 * File: DecodeInputStream.java
 *
 * ****************************************************************************
 *
 *	ADOBE CONFIDENTIAL
 *	___________________
 *
 *	Copyright 2003-2005 Adobe Systems Incorporated
 *	All Rights Reserved.
 *
 *	NOTICE: All information contained herein is, and remains the property of
 *	Adobe Systems Incorporated and its suppliers, if any. The intellectual
 *	and technical concepts contained herein are proprietary to Adobe Systems
 *	Incorporated and its suppliers and may be covered by U.S. and Foreign
 *	Patents, patents in process, and are protected by trade secret or
 *	copyright law. Dissemination of this information or reproduction of this
 *	material is strictly forbidden unless prior written permission is obtained
 *	from Adobe Systems Incorporated.
 *
 * ***************************************************************************/

package com.adobe.internal.pdftoolkit.core.filter;

import java.io.FilterInputStream;
import java.io.IOException;
import java.io.InputStream;

/**
 * DecodeInputStream
 *
 *	Parent of PDF Filters
 *  contains common code
 *
 * Copyright (C) 1996-2005 Adobe Systems Incorporated

 Modification log:
 6/10/96 McCreight -- initial version.
 7/28/97 Kaufman -- Acrobat specific version
 9/26/97 Kaufman -- reintroduce buffered version from McCreight's code
*/
abstract public class DecodeInputStream extends FilterInputStream
{
	// Variables
	protected boolean useInMark = false;
	protected byte    inBuf[] = null;
	protected int     inCount = 0;    // total valid bytes in input buf
	protected int     inPos = 0;      // next char index in input buf
	protected int     inTotal = 0;
	protected byte    outBuf[] = null;
	protected int     outCount = 0;   // total valid bytes in output buf
	protected int     outPos = 0;     // next char index in output buf
	protected int     outHeadroom = 0;
	protected int     markPos = -1;   // -1 if no mark
	protected int     markLimit = 0;
	protected FilterParams diparams = null;
	protected IOException pendingException = null;
	protected boolean avoidOverrun = false;
	protected boolean closeSource = true;
	protected boolean pendingEOF = false;   // the Filter subclass must set this at EOF!

	// Public contructors

	public DecodeInputStream(InputStream in, int inSize, int outSize, int outHeadroom, FilterParams diparams)
	{
		super(in);
		if (diparams == null)
			diparams = new FilterParams();

		this.diparams = diparams;
		this.outHeadroom = outHeadroom;
		outBuf = new byte[outSize + outHeadroom + 1];

		if (diparams.containsKey(FilterParams.AvoidInOverrun_K))
			avoidOverrun = ((Boolean)diparams.get(FilterParams.AvoidInOverrun_K)).booleanValue();
		if (diparams.containsKey(FilterParams.CloseSource_K))
			closeSource = ((Boolean)diparams.get(FilterParams.CloseSource_K)).booleanValue();

		/* Share the buffer between streams */
		if (in instanceof DecodeInputStream) {
			inBuf = ((DecodeInputStream)in).outBuf;
			useInMark = false;
		} else {
			if (avoidOverrun) {
				useInMark = in.markSupported();
				inBuf = new byte[useInMark ? inSize : 1];
			} else
				inBuf = new byte[inSize];
		}
	}

	public DecodeInputStream(InputStream in, int outHeadroom, FilterParams p)
	{
		this(in, 2048, 2048, outHeadroom, p);
	}

	public DecodeInputStream(InputStream  in, FilterParams p)
	{
		this(in, 2048, 2048, 128, p);
	}

	public DecodeInputStream(InputStream  in)
	{
		this(in, null);
	}

	public abstract void fill();

	// See BufferedInputStream.java for rationale.

	protected void doFill()
	{
		if (pendingEOF || (pendingException != null))
			return;
		makeBufferSpace();
		fill();
	}

	protected boolean fillInputBuffer()
	{
		if (inCount <= inPos) {
			if (pendingEOF || pendingException != null)
				return false;
			if (useInMark)
				in.mark(inBuf.length);
			try {
				if (in instanceof DecodeInputStream) {
					DecodeInputStream din = (DecodeInputStream)in;

					/* Fill shared buffer */
					din.outPos = inPos;
					din.doFill();
					inBuf = din.outBuf;
					inPos = din.outPos;
					inCount = din.outCount;
				} else {
					inPos = 0;
					inTotal += inCount;
					inCount = in.read(inBuf, inPos, inBuf.length);
				}
			} catch (IOException e) {
				pendingException = e;
				return false;
			}
			if (inCount <= inPos) {
				return false;
			}
		}
		return true;
	}

	protected void makeBufferSpace()
	{
		// Called with outPos == outCount.
		while ((0 <= markPos) && (outBuf.length <= (outPos + outHeadroom))) {
			// We must retain the bytes in [markPos..markPos+markLimit)
			// but there isn't sufficient headroom in the buffer above
			// outPos to add new data.
			if (0 < markPos){
				// slide buffer downwards, with markPos going to 0
				outPos -= markPos;
				System.arraycopy(outBuf, markPos, outBuf, 0, outPos);
				markPos = 0;
			} else {
				long markLimitPlusHeadroom = Math.min((long)markLimit + outHeadroom + 1, Integer.MAX_VALUE);
				if (markLimitPlusHeadroom <= outBuf.length) {
					// moving beyond reserved area
					markPos = -1; // invalidate mark
				} else {
					// larger buffer, please
					if(!increaseBufferSize(markLimitPlusHeadroom))
						break;
				}
			}
		}
		if (markPos < 0)
			outPos = 0;     // no valid mark; throw away buffer
		outCount = outPos;
	}

	protected boolean increaseBufferSize(long markLimitPlusHeadroom){
		int newSize = (2 * outBuf.length) - (outHeadroom + 1);
		if (markLimitPlusHeadroom < newSize)
			newSize = (int) markLimitPlusHeadroom;
		byte newOutBuf[] = new byte[newSize];
		System.arraycopy(outBuf, 0, newOutBuf, 0, outPos);
		outBuf = newOutBuf;
		return true;
	}
	// Public instance methods

	@Override
	public int read()
	{
		if (outCount <= outPos) {
			doFill();
			if (outCount <= outPos)
				return -1;
		}
		return outBuf[outPos++] & 0xFF;
	}

	@Override
	public int read(byte b[], int off, int len)
		throws IOException
	{
		if (b == null)
			return (int)skip(len);

		if (0 < len) {
			if (outCount <= outPos) {
				doFill();
				if (outCount <= outPos) {
					if (pendingException == null)
						return -1;
					else
						throw pendingException;
				}
			}
			int xfer = outCount - outPos;
			if (len > xfer)
				len = xfer;
			System.arraycopy(outBuf, outPos, b, off, len);
			outPos += len;
		}
		return len;
	}

	@Override
	public long skip(long len)
		throws IOException
	{
		if (0 < len) {
			if (outCount <= outPos) {
				doFill();
				if (outCount <= outPos) {
					if (pendingException == null)
						return -1;
					else
						throw pendingException;
				}
			}
			int xfer = outCount - outPos;
			if (len > xfer)
				len = xfer;
			outPos += (int)len;
		}
		return len;
	}

	@Override
	public int available()
	{
		if (outCount <= outPos)
			doFill();
		return (outCount - outPos);
	}

	@Override
	public void mark(int readlimit)
	{
		markLimit = readlimit;
		markPos = outPos;
	}

	@Override
	public void reset()
		throws IOException
	{
		if (markPos < 0)
			throw new FilterException("DecodeInput reset to invalid mark");
		else
			outPos = markPos;
	}

	@Override
	public boolean markSupported() { return true; }

	/* releaseUnconsumedInput
	 *
	 * We need to pull input up through the EOF mark, then release anything
	 * not used back to the (ordinary) input stream. Because all filter streams
	 * have an EOF marker, it is only necessary to release on the earliest one
	 */
	public void releaseUnconsumedInput()
	{
		if (in instanceof DecodeInputStream) {
			((DecodeInputStream)in).releaseUnconsumedInput();
			return;
		}
		try {
			do {
				if (skip(Long.MAX_VALUE) < 0)
					pendingEOF = true;
			} while (!pendingEOF && pendingException == null);

			if (useInMark && (inPos < inCount)) {
				// give back unused input
				in.reset();
				in.skip(inPos);
				in.mark(0);
			}
		} catch (IOException e) {
			if (pendingException == null)
				pendingException = e;
		}

		inCount = 0;
		inPos = 0;
	}

	public int getInPos()
	{
		if (in instanceof DecodeInputStream)
			return ((DecodeInputStream)in).getInPos();
		return inTotal + inPos;
	}
	
	@Override
	public void close()
		throws IOException
	{
		if (avoidOverrun)
			releaseUnconsumedInput();
		if (closeSource)
			super.close();
	}
}
