/* ****************************************************************************
 *
 *	File: SecurityHandler.java
 *
 * ****************************************************************************
 *
 *	ADOBE CONFIDENTIAL
 *	___________________
 *
 *	Copyright 2004-2005 Adobe Systems Incorporated
 *	All Rights Reserved.
 *
 *	NOTICE: All information contained herein is, and remains the property of
 *	Adobe Systems Incorporated and its suppliers, if any. The intellectual
 *	and technical concepts contained herein are proprietary to Adobe Systems
 *	Incorporated and its suppliers and may be covered by U.S. and Foreign
 *	Patents, patents in process, and are protected by trade secret or
 *	copyright law. Dissemination of this information or reproduction of this
 *	material is strictly forbidden unless prior written permission is obtained
 *	from Adobe Systems Incorporated.
 *
 * ***************************************************************************/
package com.adobe.internal.pdftoolkit.core.securityframework;

import java.util.Map;


import com.adobe.internal.pdftoolkit.core.exceptions.PDFSecurityAuthorizationException;
import com.adobe.internal.pdftoolkit.core.exceptions.PDFSecurityConfigurationException;
import com.adobe.internal.pdftoolkit.core.permissionprovider.PermissionProvider;


/**
 * <code>SecurityHandler</code> performs the main actions required to decrypt or encrypt a PDF document. 
 * In the case of decryption, the <code>SecurityHandler</code> authenticates the right to decrypt 
 * the document. For example, password-based <code>SecurityHandler</code> validitates 
 * the password in the process of authentication.  </P>
 * 
 * <P><code>SecurityHandler</code> also delivers (on decryption) or determines (on encryption) encryption
 * permissions. For the password-based security handlers, this is the content of the 
 * <B>/P</B> entry in the <B>Encrypt</B> dictionary; 
 * the public-key-based security handlers get the encryption permissions from the proper PKCS#7
 * object in the <B>Recipients</B> array in the <B>Encrypt</B> dictionary. Other security 
 * handlers implementations may get/set encryption permissions as they deem appropriate. 
 * Therefore the <code>SecurityHandler</code> interface contains the method that returns 
 * <CODE>PermissionsProvider</CODE> that handles encryption permissions.</P>
 * 
 * <P><code>SecurityHandler</code> declares the interfaces for a specific security 
 * handler in a <code>PDFDocument</code>. </P>
 * 
 * <P>The security handler provides the following functionality:  </P>
 * <UL> 
 * <li>Authenticates the access rights to an encrypted PDF document.</li>
 * <li>Computes the base encryption key.</li>
 * <li>Computes the object-specific encryption key for each PDF object (String or Stream).</li>
 * <li>Uses the object-specific encryption key to encrypt or decrypt (String or Stream) content.</li>
 * <LI>All Security Handler implementations must support Crypt filters listed in the <B>Encrypt</B> 
 * dictionary. Specifically, they must support <B>Identity</B> Crypt filter. </LI>
 * </UL>
 * 
 * <P>The <i>PDF Java ToolKit</i> provides the following password-based security handler.</P> 
 * <UL>
 *    <LI><B>Standard </b>This security handler provides support for RC4 and AES 
 * encryption technologies. </LI>
 * </UL>
 * 
 * <P>The names for other security handlers can be registered by using the 
 * procedure described in the <i>PDF Reference</i>,  
 * <a href="http://partners.adobe.com/public/developer/en/pdf/PDFReference16.pdf#page=959">Appendix E</a>.
 * </P>
 */
public interface SecurityHandler 
{
	/**
	 * Instantiates an encryption handler that performs the actual
	 * encryption/decryption operations with the base encryption key
	 * calculated by this security handler.
	 * To implement this, Security Handler should authenticate the access rights to
	 * the encrypted document before it instantiates the encryption handler.
	 *  
	 * @param cryptName the name of the Crypt Filter
	 * @param encryptParams the Encryption Parameters Map containing encryption
	 * parameters for encryption. Depending on the implementation it may or may
	 * not be the same Encryption Parameters Map used in decryption.
	 * <code>SecurityHandler</code> may cache or discard it.
	 * 
	 * @param docId the first Document ID string in the PDF documents trailer.
	 * @return	<code>EncryptionHandler</code> encryption handler that performs encryption/decryption for this  
	 * security handler. This handler may be cached by the Security Handler
	 * for reuse as required.
	 * @throws PDFSecurityAuthorizationException
     * @throws PDFSecurityConfigurationException
	 * @see SecurityHandler#authenticate(Map, byte[]) 
	 */
	public EncryptionHandler getEncryptionHandler(String cryptName, Map encryptParams, byte[] docId) 
    throws PDFSecurityAuthorizationException, PDFSecurityConfigurationException;
	
	/**
	 * Get the encryption parameters specified by this Security Handler
	 * This can be either the original Encryption Parameters Map that were 
	 * passed in the {@link com.adobe.internal.pdftoolkit.core.securityframework.SecurityHandler#getEncryptionHandler(String, Map, byte[])}
	 * call with some entries possibly changed.
	 * It could also be a completely new Map created in the SecurityHandler and
	 * used to encrypt content. </P>
	 * Security Handler should not change Encryption Parameters Map used to decrypt
	 * the content
	 * <P>In instances of the Security Handler for decryption this method may return 
	 * <code>null</code> because no encryption parameters were saved. </P> 
	 * @return Map that contains encryption parameters.
	 */
	public Map getEncryptParameters();

	/**
	 * Authenticates the right to decrypt and access the document.
	 * <P>This method is usually called for the default Security Handler 
	 * by the Security Support implementation to ensure the ability to 
	 * process PDF document. Because particular streams 
	 * may use different Security Handlers for  encryption (or decryption), 
	 * the authentication of these Security Handlers
	 * may happen, depending on the implementation, in the 
	 * {@link com.adobe.internal.pdftoolkit.core.securityframework.SecurityHandler#getEncryptionHandler(String, Map, byte[])} call. </P>
	 * @param params the Encryption Parameters Map containing encryption
	 * parameters for encryption. Depending on the implementation it may or may
	 * not be the same Encryption Parameters Map used in decryption.</P>
	 * <P>SecurityHandler may cache or discard the parameters map. </P>
	 * @param docID the first Document ID string in the PDF documents trailer.
     * @throws PDFSecurityAuthorizationException
     * @throws PDFSecurityConfigurationException
	 */
	public boolean authenticate(Map params, byte[] docID) 
    throws PDFSecurityAuthorizationException, PDFSecurityConfigurationException;	
	/**
	 *  Fetch the permission provider that specifies encryption permissions associated with this security handler.
	 */
	public PermissionProvider getPermissionProvider();

	/**
	 * Authenticates the right to decrypt and access the document by plugging in
	 * the decrypted state of the document.
	 * @param params the Encryption Parameters Map containing encryption
	 * parameters for encryption. Depending on the implementation it may or may
	 * not be the same Encryption Parameters Map used in decryption.</P>
	 * <P>SecurityHandler may cache or discard the parameters map. </P>
	 * @param docID the first Document ID string in the PDF documents trailer.
	 * @param decryptedState Decrypted state of the document 
     * @throws PDFSecurityAuthorizationException
     * @throws PDFSecurityConfigurationException
	 */
	public boolean authenticate(Map params, byte[] docID, DecryptedState decryptedState)
	throws PDFSecurityConfigurationException, PDFSecurityAuthorizationException;
	
	/**
	 * Returns decrypted state of the document. If authenticate method has not
	 * been called, then this method returns null. 
	 * @return DecryptedState
	 */
	public DecryptedState getDecryptedState();

}
